<?php
/**
 * Plugin Name: POS OfiSoftWeb
 * Plugin URI: https://ofisoftweb.com
 * Description: Sistema POS para WooCommerce diseñado para facilitar la gestión de ventas, la edición de pedidos, la impresión de tickets personalizados y la generación de tickets regalo y proformas sin impacto en el inventario.
 * Version: 2.8.0
 * Author: OfiSoftWeb
 * Author URI: https://ofisoftweb.com
 * Requires at least: 6.0
 * Tested up to: 6.6
 * Requires PHP: 7.4
 * WC requires at least: 7.0
 * WC tested up to: 9.0
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: pos-ofisoftweb
 * Company: OfiSoftWeb
 * Support Email: Plugins@ofisoftweb.com
 */
if (!defined('ABSPATH')) exit;

// Declaración de compatibilidad con características de WooCommerce (HPOS / Custom Order Tables)
add_action('before_woocommerce_init', function () {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

define('POS_TIENDA_QUERYVAR','pos_tienda');
define('POS_TIENDA_SALES_QUERYVAR','pos_tienda_sales');
define('POS_TIENDA_REPORTS_QUERYVAR','pos_tienda_reports');
define('POS_TIENDA_CLIENTS_QUERYVAR','pos_tienda_clients');
define('POS_TIENDA_DAILY_REPORT_QUERYVAR','pos_tienda_daily_report');

/**
 * Marcar pedidos creados desde el POS para que WooCommerce los identifique como origen POS.
 *
 * 1) Columna "Origen" (Atribución de pedidos): escribimos metadatos de atribución para que no salga "Desconocido".
 * 2) created_via: añade trazabilidad interna.
 */
function pos_tienda_mark_order_as_pos($order){
    if (!is_object($order) || !method_exists($order, 'update_meta_data')) {
        return;
    }

    // created_via (trazabilidad)
    if (method_exists($order, 'set_created_via')) {
        $order->set_created_via('pos-ofisoftweb');
    }

    // Atribución (columna "Origen" en WooCommerce > Pedidos)
    // Usamos UTM para que WooCommerce muestre "Fuente: POS OfiSoftWeb" en lugar de "Desconocido".
    $now = time();
    $order->update_meta_data('_wc_order_attribution_source_type', 'utm');
    $order->update_meta_data('_wc_order_attribution_utm_source', 'POS OfiSoftWeb');
    $order->update_meta_data('_wc_order_attribution_utm_medium', 'pos');
    $order->update_meta_data('_wc_order_attribution_utm_campaign', 'POS');
    $order->update_meta_data('_wc_order_attribution_referrer', 'pos-ofisoftweb');
    $order->update_meta_data('_wc_order_attribution_session_entry', 'pos-ofisoftweb');
    $order->update_meta_data('_wc_order_attribution_session_start_time', $now);
    $order->update_meta_data('_wc_order_attribution_user_agent', 'POS OfiSoftWeb');
}

/* ===== Idioma (ES/EN) ===== */
function pos_tienda_get_lang(){
    $lang = get_option('pos_tienda_lang', 'es');
    $lang = is_string($lang) ? strtolower(trim($lang)) : 'es';
    return ($lang === 'en') ? 'en' : 'es';
}

function pos_tienda_i18n($es, $en){
    return (pos_tienda_get_lang() === 'en') ? $en : $es;
}

function pos_tienda_i18n_map(){
    return array(
        // Admin / navegación
        'Abrir POS' => 'Open POS',
        'Ventas POS' => 'POS Sales',
        'Informes POS' => 'POS Reports',
        'Clientes POS' => 'POS Clients',
        'Informe diario' => 'Daily report',
        'Idioma' => 'Language',
        'Aplicar y guardar' => 'Apply & Save',

        // POS header
        'Datos emisor' => 'Issuer data',
        'Reset numeración' => 'Reset numbering',
        'Imprimir' => 'Print',
        'Cerrar' => 'Close',

        // POS principal
        'Productos' => 'Products',
        'Carrito' => 'Cart',
        'Buscar por título, SKU o descripción (unificado)...' => 'Search by title, SKU or description…',
        'Producto' => 'Product',
        'Precio (€)' => 'Price (€)',
        'Unidades' => 'Qty',
        'Título del producto' => 'Product title',
        'SKU (opcional)' => 'SKU (optional)',
        'Precio (ej. 14,99)' => 'Price (e.g. 14.99)',
        'Añadir' => 'Add',
        'Cobrar' => 'Charge',
        'Ventas' => 'Sales',
        'Informes' => 'Reports',
        'Clientes' => 'Clients',
        'Impuestos' => 'Taxes',
        'Anotaciones' => 'Notes',
        'Observaciones del pedido (se imprimen)' => 'Order notes (printed)',
        'Dto. (%)' => 'Disc. (%)',

        // Cobro
        'Cobro' => 'Payment',
        'Método de pago' => 'Payment method',
        'Efectivo' => 'Cash',
        'Tarjeta' => 'Card',
        'Efectivo recibido' => 'Cash received',
        'Cambio' => 'Change',
        'Tipo de documento' => 'Document type',
        'Ticket' => 'Ticket',
        'Proforma' => 'Proforma',
        'Factura' => 'Invoice',
        'Cliente (buscar)' => 'Customer (search)',
        'Nombre, Nº cliente, NIF, empresa, teléfono o email' => 'Name, customer #, VAT, company, phone or email',
        'Empresa' => 'Company',
        'Nombre y apellidos' => 'Full name',
        'NIF' => 'VAT/Tax ID',
        'Dirección' => 'Address',
        'Email' => 'Email',
        'Teléfono' => 'Phone',
        'Cancelar' => 'Cancel',
        'Anterior' => 'Back',
        'Siguiente' => 'Next',

        // Impuestos modal
        'Nombre del impuesto' => 'Tax name',
        'Porcentaje (%)' => 'Rate (%)',
        'Ejemplo: 21 = 21% (se calcula como 1,21)' => 'Example: 21 = 21% (calculated as 1.21)',
        'Guardar' => 'Save',

        // Informes
        'Desde' => 'From',
        'Hasta' => 'To',
        'Método' => 'Method',
        'Todos' => 'All',
        'Buscar' => 'Search',
        'Inicio' => 'Home',
        'Nota: Pulsa "Inicio" para volver al POS.' => 'Note: Press “Home” to go back to the POS.',
        'Total POS' => 'POS total',
        'Pedidos' => 'Orders',
        'No se pudo cargar' => 'Could not load',

        // Clientes
        'Nuevo cliente' => 'New customer',
        'Buscar cliente (nombre, empresa, NIF, teléfono, email)' => 'Search customer (name, company, VAT, phone, email)',
        'Acciones' => 'Actions',
        'Editar' => 'Edit',
        'Eliminar' => 'Delete',
        'Todavía no hay clientes guardados.' => 'No customers saved yet.',
        'Cliente POS' => 'POS Customer',
        'Nº cliente' => 'Customer #',
        'Notas' => 'Notes',
        'Nuevo' => 'New',
        'Guardar cliente' => 'Save customer',

        // Ventas
        'Ventas POS' => 'POS Sales',
        'Todas' => 'All',
        'Tickets' => 'Tickets',
        'Proformas' => 'Proformas',
        'Facturas' => 'Invoices',
        'Buscar venta (código, documento, fecha, total, estado)' => 'Search sale (code, document, date, total, status)',
        'Código' => 'Code',
        'Fecha' => 'Date',
        'Total' => 'Total',
        'Estado' => 'Status',
        'IMPRIMIR' => 'PRINT',
        'MODIFICAR' => 'EDIT',
        'EDITAR EN POS' => 'EDIT IN POS',
        'ELIMINAR' => 'DELETE',
        'TICKET REGALO' => 'GIFT RECEIPT',
        'Editar pedido' => 'Edit order',
        'Precio' => 'Price',
        'Uds' => 'Qty',

        // Diario
        'Incluye tickets y facturas del POS. Excluye proformas.' => 'Includes POS tickets and invoices. Excludes proformas.',
        'Nº pedidos' => 'Orders',
        'Total efectivo' => 'Cash total',
        'Total tarjeta' => 'Card total',
        'No hay ventas POS hoy.' => 'No POS sales today.',
        'Impreso:' => 'Printed:',

        // Mensajes frecuentes
        'No hay stock suficiente' => 'Not enough stock',
        'Título requerido' => 'Title is required',
        'Precio inválido' => 'Invalid price',
        'Efectivo insuficiente' => 'Insufficient cash',
        'Completa los datos del receptor' => 'Please complete the recipient details',
        'Error al cobrar' => 'Payment error',
        'Error de conexión' => 'Connection error',
        'Impuestos guardados' => 'Taxes saved',
        'No se pudo guardar' => 'Could not save',
        'Cliente guardado' => 'Customer saved',
        'No se pudo guardar el cliente' => 'Could not save the customer',
        'ID de cliente no válido.' => 'Invalid customer ID.',
        '¿Eliminar este cliente?' => 'Delete this customer?',
        'No se pudo eliminar el cliente' => 'Could not delete the customer',
        'Eliminar pedido POS (sin reponer stock)?' => 'Delete POS order (without restoring stock)?',
        'No se pudo eliminar' => 'Could not delete',
    );
}

function pos_tienda_print_i18n_script(){
    $lang = pos_tienda_get_lang();
    $locale = ($lang === 'en') ? 'en-IE' : 'es-ES';
    $map = pos_tienda_i18n_map();
    ?>
    <script>
    window.POS_LANG = <?php echo wp_json_encode($lang); ?>;
    window.POS_LOCALE = <?php echo wp_json_encode($locale); ?>;
    window.POS_I18N = <?php echo wp_json_encode($map); ?>;
    (function(){
        var lang = window.POS_LANG || 'es';
        function tr(msg){
            msg = String(msg === null || msg === undefined ? '' : msg);
            if (lang !== 'en') return msg;
            var map = window.POS_I18N || {};
            if (map[msg]) return map[msg];
            if (msg.indexOf('Informe diario ·') === 0) return msg.replace('Informe diario ·','Daily report ·');
            if (msg.indexOf('Informe diario') === 0) return msg.replace('Informe diario','Daily report');
            if (msg.indexOf('Total sin ') === 0) return msg.replace('Total sin ','Total excl. ');
            if (msg.indexOf('Total ') === 0 && msg.indexOf(' incuido') !== -1) return msg.replace('Total ','Total ').replace(' incuido',' included');
            if (msg.indexOf('Método de pago: ') === 0) return msg.replace('Método de pago: ','Payment method: ');
            if (msg.indexOf('Cantidad entregada: ') === 0) return msg.replace('Cantidad entregada: ','Cash received: ');
            if (msg.indexOf('Cambio: ') === 0) return msg.replace('Cambio: ','Change: ');
            if (msg.indexOf('Fecha: ') === 0) return msg.replace('Fecha: ','Date: ');
            if (msg.indexOf('Cliente') === 0) return msg.replace('Cliente','Customer');
            return msg;
        }
        try {
            var _alert = window.alert;
            window.alert = function(m){ return _alert(tr(m)); };
            var _confirm = window.confirm;
            window.confirm = function(m){ return _confirm(tr(m)); };
        } catch(e) {}
        function apply(){
            if (lang !== 'en') return;
            try {
                document.querySelectorAll('[placeholder]').forEach(function(el){
                    var p = el.getAttribute('placeholder') || '';
                    var np = tr(p);
                    if (np !== p) el.setAttribute('placeholder', np);
                });
                document.querySelectorAll('option').forEach(function(o){
                    var t = (o.textContent || '').trim();
                    var nt = tr(t);
                    if (nt !== t) o.textContent = nt;
                });
                var w = document.createTreeWalker(document.body, NodeFilter.SHOW_TEXT, null, false);
                var nodes = [];
                while (w.nextNode()) nodes.push(w.currentNode);
                nodes.forEach(function(n){
                    var raw = n.nodeValue;
                    if (!raw) return;
                    var trimmed = raw.trim();
                    if (!trimmed) return;
                    var repl = tr(trimmed);
                    if (repl !== trimmed){
                        n.nodeValue = raw.replace(trimmed, repl);
                        return;
                    }
                    if (trimmed.indexOf('Informe diario ·') === 0){
                        var r2 = tr(trimmed);
                        if (r2 !== trimmed) n.nodeValue = raw.replace(trimmed, r2);
                    }
                });
                document.documentElement.lang = lang;
            } catch(e) {}
        }
        if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', apply);
        else apply();
    })();
    </script>
    <?php
}


register_activation_hook(__FILE__, function(){
    add_rewrite_tag('%'.POS_TIENDA_QUERYVAR.'%','1');
    add_rewrite_tag('%'.POS_TIENDA_SALES_QUERYVAR.'%','1');
    add_rewrite_tag('%'.POS_TIENDA_REPORTS_QUERYVAR.'%','1');
    add_rewrite_tag('%'.POS_TIENDA_CLIENTS_QUERYVAR.'%','1');
    add_rewrite_tag('%'.POS_TIENDA_DAILY_REPORT_QUERYVAR.'%','1');
    add_rewrite_rule('^pos-tienda/?$','index.php?'.POS_TIENDA_QUERYVAR.'=1','top');
    add_rewrite_rule('^pos-tienda-ventas/?$','index.php?'.POS_TIENDA_SALES_QUERYVAR.'=1','top');
    add_rewrite_rule('^pos-tienda-informes/?$','index.php?'.POS_TIENDA_REPORTS_QUERYVAR.'=1','top');
    add_rewrite_rule('^pos-tienda-clientes/?$','index.php?'.POS_TIENDA_CLIENTS_QUERYVAR.'=1','top');
    add_rewrite_rule('^pos-tienda-informe-diario/?$','index.php?'.POS_TIENDA_DAILY_REPORT_QUERYVAR.'=1','top');
    flush_rewrite_rules();
});

register_deactivation_hook(__FILE__, function(){ flush_rewrite_rules(); });
add_filter('query_vars', function($v){ $v[]=POS_TIENDA_QUERYVAR; $v[]=POS_TIENDA_SALES_QUERYVAR; $v[]=POS_TIENDA_REPORTS_QUERYVAR; $v[]=POS_TIENDA_CLIENTS_QUERYVAR; $v[]=POS_TIENDA_DAILY_REPORT_QUERYVAR; return $v; });

add_action('admin_menu', function(){
    add_menu_page('POS OfiSoftWeb','POS OfiSoftWeb','manage_woocommerce','pos-tienda-launch', function(){
        if (!current_user_can('manage_woocommerce')) { return; }

        // Guardar idioma
        if (isset($_POST['pos_tienda_lang'])) {
            check_admin_referer('pos_tienda_save_lang');
            $new = sanitize_text_field( wp_unslash($_POST['pos_tienda_lang']) );
            $new = ($new === 'en') ? 'en' : 'es';
            update_option('pos_tienda_lang', $new, false);
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html( pos_tienda_i18n('Idioma guardado.','Language saved.') ) . '</p></div>';
        }

        $lang = pos_tienda_get_lang();

        $pos     = esc_url(add_query_arg(POS_TIENDA_QUERYVAR,1,home_url('/')));
        $sales   = esc_url(add_query_arg(POS_TIENDA_SALES_QUERYVAR,1,home_url('/')));
        $reports = esc_url(add_query_arg(POS_TIENDA_REPORTS_QUERYVAR,1,home_url('/')));
        $clients = esc_url(add_query_arg(POS_TIENDA_CLIENTS_QUERYVAR,1,home_url('/')));
        $daily   = esc_url(add_query_arg(POS_TIENDA_DAILY_REPORT_QUERYVAR,1,home_url('/')));

        echo '<div class="wrap">';
        echo '<h1>POS OfiSoftWeb</h1>';

        // Selector ES/EN
        echo '<form method="post" style="margin:12px 0 16px; padding:12px 14px; border:1px solid #dcdcde; background:#fff; border-radius:10px; display:flex; gap:10px; align-items:center; flex-wrap:wrap; max-width:640px;">';
        wp_nonce_field('pos_tienda_save_lang');
        echo '<strong style="min-width:120px">' . esc_html( pos_tienda_i18n('Idioma','Language') ) . '</strong>';
        echo '<select name="pos_tienda_lang">'
            . '<option value="es"' . selected($lang,'es',false) . '>Español</option>'
            . '<option value="en"' . selected($lang,'en',false) . '>English</option>'
            . '</select>';
        echo '<button class="button button-primary" type="submit">' . esc_html( pos_tienda_i18n('Aplicar y guardar','Apply & Save') ) . '</button>';
        echo '<span style="opacity:.75">' . esc_html( pos_tienda_i18n('Este idioma se aplica a todo el POS.','This language applies to the whole POS.') ) . '</span>';
        echo '</form>';

        echo '<p>';
        echo '<a class="button button-primary" target="_blank" href="'.$pos.'">'.esc_html(pos_tienda_i18n('Abrir POS','Open POS')).'</a> ';
        echo '<a class="button" target="_blank" href="'.$sales.'">'.esc_html(pos_tienda_i18n('Ventas POS','POS Sales')).'</a> ';
        echo '<a class="button" target="_blank" href="'.$reports.'">'.esc_html(pos_tienda_i18n('Informes POS','POS Reports')).'</a> ';
        echo '<a class="button" target="_blank" href="'.$clients.'">'.esc_html(pos_tienda_i18n('Clientes POS','POS Clients')).'</a> ';
        echo '<a class="button" target="_blank" href="'.$daily.'">'.esc_html(pos_tienda_i18n('Informe diario','Daily report')).'</a>';
        echo '</p>';

        // Suscripción voluntaria (soporte)
        echo '<div class="ofisoftweb-subscription" style="margin:16px 0; padding:16px 18px; border:1px solid #dcdcde; border-left:4px solid #2271b1; background:#fff; border-radius:10px; box-shadow:0 1px 2px rgba(0,0,0,.04);">';
        echo '  <div style="display:flex; gap:14px; align-items:flex-start; flex-wrap:wrap;">';
        echo '    <span class="dashicons dashicons-awards" style="font-size:34px; width:34px; height:34px; color:#2271b1;"></span>';
        echo '    <div style="flex:1 1 420px;">';
                echo '      <h2 style="margin:0 0 6px 0; font-size:18px;">' . esc_html( pos_tienda_i18n('Haz tu donación y ayúdanos a crear más plugins gratuitos','Make a donation and help us create more free plugins') ) . '</h2>';
                echo '      <p style="margin:0 0 10px 0; color:#1d2327;">' . esc_html( pos_tienda_i18n(
            'Pulsa en el enlace y haz tu donación para poder seguir construyendo plugins. Envíanos tu propuesta para nuevos plugins a Plugins@ofisoftweb.com.',
            'Click the link to make a donation so we can keep building plugins. Send your plugin ideas to Plugins@ofisoftweb.com.'
        ) ) . '</p>';
                echo '      <div style="display:flex; gap:10px; align-items:center; flex-wrap:wrap;">';
        echo '        <a class="button button-primary" href="https://ofisoftweb.com" target="_blank" rel="noopener">' . esc_html( pos_tienda_i18n('Hacer donación','Make a donation') ) . '</a>';
        echo '        <a class="button" href="mailto:Plugins@ofisoftweb.com">' . esc_html( pos_tienda_i18n('Enviar propuesta','Send proposal') ) . '</a>';
        echo '      </div>';
        echo '    </div>';
        echo '  </div>';
        echo '</div>';

        echo '</div>';

    }, 'dashicons-cart', 56);
});


function pos_tienda_css(){ ?>
<style>
html,body{height:100%;margin:0;background:#d7f7d7;font:16px/1.45 system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial;color:#000}
a{color:inherit}
.btn,.panel h2, header.pos, .table th,.table td{color:#000}
header.pos{display:flex;align-items:center;justify-content:space-between;padding:12px 16px;background:rgba(255,255,255,.7);border-bottom:1px solid #d3e9d3}
header.pos .btn{border:1px solid #d3e9d3;background:#fff;border-radius:10px;padding:8px 10px;cursor:pointer}
main.pos{display:grid;grid-template-columns:1fr 1fr;gap:14px;padding:14px;height:calc(100vh - 56px);overflow:hidden}
.panel{background:#fff;border:1px solid #d3e9d3;border-radius:16px;height:100%;display:flex;flex-direction:column;overflow:hidden}
.panel h2{margin:0;padding:12px 14px;border-bottom:1px solid #d3e9d3;font-size:16px;background:#f7fff7}
.toolbar{padding:12px 14px;display:flex;gap:10px;align-items:center;border-bottom:1px solid #d3e9d3;background:#fff}
.toolbar input[type="search"]{flex:1;padding:10px 12px;border:1px solid #d3e9d3;border-radius:10px;outline:none}
.prod-header{display:grid;grid-template-columns:56px 2fr 1fr .9fr .8fr .9fr;gap:8px;padding:8px 12px;border-bottom:1px solid #d3e9d3;background:#fff;font-size:12px}
.prod-rows{overflow:auto;padding:8px 12px;display:flex;flex-direction:column;gap:8px;flex:1 1 auto;padding-bottom:180px}
.prod-row{display:grid;grid-template-columns:56px 2fr 1fr .9fr .8fr .9fr;gap:8px;align-items:center;border:1px solid #d3e9d3;border-radius:12px;padding:8px;background:#fff}
.prod-row .title{font-weight:700;font-size:16px}
.prod-row .price{text-align:right}
.prod-row input.qty{width:100%;padding:8px 10px;border:1px solid #d3e9d3;border-radius:10px;font-size:13px}
.left-split{display:grid;grid-template-rows:3fr 1fr}
.manual-add{position:sticky;bottom:96px;background:#fff;border:1px solid #d3e9d3;padding:10px 12px;display:grid;grid-template-columns:2fr 1fr 1fr .8fr .9fr;gap:8px;align-items:center;border-radius:12px;margin:0 12px 12px}
.cart-header{display:grid;grid-template-columns:56px 1.6fr 1fr .9fr .9fr .8fr 1fr .6fr;gap:8px;padding:8px 12px;border-bottom:1px solid #d3e9d3;background:#fff;font-size:12px}
.cart-rows{overflow:auto;padding:8px 12px;display:flex;flex-direction:column;gap:8px}
.cart-row{display:grid;grid-template-columns:56px 1.6fr 1fr .9fr .9fr .8fr 1fr .6fr;gap:8px;align-items:center;border:1px solid #d3e9d3;border-radius:12px;padding:8px;background:#fff}
.cart-row .total{font-weight:700;text-align:right}
.cart-row input[type="number"], .cart-row input[type="text"]{width:100%;padding:8px 10px;border:1px solid #d3e9d3;border-radius:10px;font-size:13px}
#opciones .options-body{padding:16px;display:grid;grid-template-columns:auto auto auto auto;gap:12px}
.btn{display:inline-flex;align-items:center;gap:6px;padding:8px 10px;border:1px solid #d3e9d3;border-radius:10px;background:#fff;cursor:pointer;text-decoration:none;color:#000}
.btn.primary{background:#eaffea;font-weight:700}
.table{width:100%;border-collapse:collapse}
.table th,.table td{padding:8px;border-bottom:1px solid #d3e9d3;text-align:left;font-size:14px}
.table td.actions{display:flex;gap:8px;align-items:center;flex-wrap:nowrap}
.bad{color:#000}
.modal{position:fixed;inset:0;background:rgba(0,0,0,.45);display:none;align-items:center;justify-content:center;padding:24px;z-index:9999}
.modal .card{background:#fff;border-radius:14px;max-width:920px;width:100%;max-height:90vh;overflow:auto;border:1px solid #d3e9d3}
.modal .card header{padding:12px 16px;border-bottom:1px solid #e5efe5;font-weight:700}
.modal .card .body{padding:12px 16px;display:grid;gap:12px}
.modal .footer{display:flex;justify-content:space-between;gap:8px;padding:12px 16px;border-top:1px solid #e5efe5}
.modal .grid{display:grid;grid-template-columns:2fr 1fr .8fr 1fr .6fr;gap:12px}
.modal label{display:flex;flex-direction:column;gap:6px}
.modal input,.modal select, .modal textarea{padding:8px 10px;border:1px solid #d3e9d3;border-radius:10px;color:#000}
.modal .row input{width:100%}
.stepper{display:flex;gap:8px;align-items:center;justify-content:center;margin-bottom:8px}
.stepper .dot{width:10px;height:10px;border-radius:999px;background:#d3e9d3}
.stepper .dot.active{background:#5db55d}
#issuerModal .grid{display:block !important;}
#issuerModal label{display:grid !important; grid-template-columns:140px 1fr !important; align-items:center; gap:10px; margin:0 0 10px 0;}
#issuerModal label:last-child{margin-bottom:0;}
#issuerModal label input{grid-column:2 !important; width:100%; box-sizing:border-box;}
@media (max-width:480px){ #issuerModal label{grid-template-columns:110px 1fr !important;} }
@media print{
  header.pos, .toolbar{display:none !important;}
  main.pos{display:block !important;height:auto !important;overflow:visible !important;padding:0 !important;}
  .panel{height:auto !important;overflow:visible !important;border:none !important;}
  table{width:100% !important;border-collapse:collapse !important;}
  thead{display:table-header-group;}
  tfoot{display:table-footer-group;}
  tr{page-break-inside:avoid;}
}
</style>
<?php }

add_action('template_redirect','pos_tienda_router',0);
function pos_tienda_router(){
    $is_pos   = get_query_var(POS_TIENDA_QUERYVAR) || isset($_GET[POS_TIENDA_QUERYVAR]);
    $is_sales = get_query_var(POS_TIENDA_SALES_QUERYVAR) || isset($_GET[POS_TIENDA_SALES_QUERYVAR]);
    $is_reports = get_query_var(POS_TIENDA_REPORTS_QUERYVAR) || isset($_GET[POS_TIENDA_REPORTS_QUERYVAR]);
    $is_clients = get_query_var(POS_TIENDA_CLIENTS_QUERYVAR) || isset($_GET[POS_TIENDA_CLIENTS_QUERYVAR]);
    $is_daily_report = get_query_var(POS_TIENDA_DAILY_REPORT_QUERYVAR) || isset($_GET[POS_TIENDA_DAILY_REPORT_QUERYVAR]);
    if (!$is_pos && !$is_sales && !$is_reports && !$is_clients && !$is_daily_report) return;

if (!is_user_logged_in()) { auth_redirect(); exit; }
    if (!headers_sent()){ nocache_headers(); status_header(200); }
    if (!defined('DONOTCACHEPAGE')) define('DONOTCACHEPAGE', true);
    show_admin_bar(false);
    add_action('wp_head','pos_tienda_css');

    ?><!doctype html><html lang="<?php echo esc_attr(pos_tienda_get_lang()); ?>"><head><meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">
    <title>POS OfiSoftWeb</title><?php wp_head(); ?></head><body>
    <?php pos_tienda_print_i18n_script(); ?>
    <header class="pos">
      <h1><?php echo $is_daily_report ? 'Informe diario' : 'POS OfiSoftWeb'; ?></h1>
      <?php if ($is_pos): ?>
        <button id="btnIssuer" class="btn">Datos emisor</button>
        <button id="btnDocnum" class="btn">Reset numeración</button>
      <?php elseif ($is_daily_report): ?>
        <button class="btn" onclick="try{print()}catch(e){}">Imprimir</button>
        <button class="btn" onclick="try{close()}catch(e){}">Cerrar</button>
      <?php endif; ?>
    </header>
    <?php if ($is_daily_report): ?>
      <?php if (!class_exists('WooCommerce')){ echo '<main style="padding:16px"><div class="panel"><h2>Error</h2><div class="toolbar">WooCommerce es requerido.</div></div></main>'; wp_footer(); ?></body></html><?php exit; } ?>
      <?php
        $tz = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone(get_option('timezone_string') ?: 'UTC');
        $start = new DateTime('today', $tz); $start->setTime(0,0,0);
        $end   = new DateTime('today', $tz); $end->setTime(23,59,59);
        $start_ts = $start->getTimestamp();
        $end_ts   = $end->getTimestamp();

        $args = array(
          'type'=>'shop_order','limit'=>-1,'orderby'=>'date','order'=>'ASC','status'=>'any','return'=>'objects',
          'meta_query'=>array(array('key'=>'_pos_origin','value'=>'pos_tienda','compare'=>'='))
        );
        try { $q = new WC_Order_Query($args); $orders = $q->get_orders(); } catch (Throwable $e){ $orders = array(); }

        $rows = array(); $tAll = 0; $tCash = 0; $tCard = 0; $cnt = 0;
        foreach ($orders as $order){
          $created = $order->get_date_created();
          $ts = $created ? $created->getTimestamp() : 0;
          if ($ts < $start_ts || $ts > $end_ts) continue;

          $doc_type = $order->get_meta('_pos_doc_type', true);
          if ($doc_type === 'P') continue; // excluye proformas

          $pm_wc   = $order->get_payment_method();
          $pm_meta = $order->get_meta('_pos_payment_method', true);
          $is_cash = ($pm_wc === 'pos-cash') || ($pm_meta === 'cash') || ($pm_meta === 'pos-cash');
          $is_card = ($pm_wc === 'pos-card') || ($pm_meta === 'card') || ($pm_meta === 'pos-card');
          $label   = $is_cash ? 'Efectivo' : ($is_card ? 'Tarjeta' : 'Desconocido');

          $total = (float) $order->get_total();
          $code  = $order->get_meta('_pos_doc_code', true) ?: $order->get_id();
          $time  = $created ? $created->date_i18n('H:i') : '';

          $rows[] = array(
            'code'=>$code,
            'time'=>$time,
            'method'=>$label,
            'total'=>$total,
            'status'=>$order->get_status(),
          );

          $tAll += $total; $cnt++;
          if ($is_cash) $tCash += $total;
          elseif ($is_card) $tCard += $total;
        }
        $today_label = function_exists('wp_date') ? wp_date('d/m/Y') : date('d/m/Y');
      ?>
      <main style="padding:16px;max-width:980px;margin:0 auto">
        <div class="panel" style="padding:14px">
          <h2 style="border:none;background:none;padding:0;margin:0 0 6px 0">Informe diario · <?php echo esc_html($today_label); ?></h2>
          <div style="font-size:13px;opacity:.85;margin-bottom:10px">Incluye tickets y facturas del POS. Excluye proformas.</div>

          <table class="table" style="margin-bottom:12px">
            <tbody>
              <tr><th style="width:260px">Nº pedidos</th><td><?php echo (int)$cnt; ?></td></tr>
              <tr><th>Total</th><td><?php echo html_entity_decode(wp_strip_all_tags(wc_price($tAll)), ENT_QUOTES, 'UTF-8'); ?></td></tr>
              <tr><th>Total efectivo</th><td><?php echo html_entity_decode(wp_strip_all_tags(wc_price($tCash)), ENT_QUOTES, 'UTF-8'); ?></td></tr>
              <tr><th>Total tarjeta</th><td><?php echo html_entity_decode(wp_strip_all_tags(wc_price($tCard)), ENT_QUOTES, 'UTF-8'); ?></td></tr>
            </tbody>
          </table>

          <table class="table">
            <thead>
              <tr>
                <th>Código</th>
                <th>Hora</th>
                <th>Método</th>
                <th style="text-align:right">Total</th>
                <th>Estado</th>
              </tr>
            </thead>
            <tbody>
              <?php if (empty($rows)): ?>
                <tr><td colspan="5">No hay ventas POS hoy.</td></tr>
              <?php else: foreach($rows as $r): ?>
                <tr>
                  <td><?php echo esc_html($r['code']); ?></td>
                  <td><?php echo esc_html($r['time']); ?></td>
                  <td><?php echo esc_html($r['method']); ?></td>
                  <td style="text-align:right"><?php echo html_entity_decode(wp_strip_all_tags(wc_price($r['total'])), ENT_QUOTES, 'UTF-8'); ?></td>
                  <td><?php echo esc_html($r['status']); ?></td>
                </tr>
              <?php endforeach; endif; ?>
            </tbody>
          </table>

          <div style="margin-top:12px;font-size:12px;opacity:.8">Impreso: <?php echo esc_html(function_exists('wp_date') ? wp_date('H:i:s') : date('H:i:s')); ?></div>
        </div>
      </main>
      <script>addEventListener('load',()=>{ setTimeout(()=>{ try{print()}catch(e){} }, 150); setTimeout(()=>{ try{close()}catch(e){} }, 900); });</script>
      <?php wp_footer(); ?></body></html><?php exit; ?>
    <?php endif; ?>
    <?php if ($is_pos): ?>
        <?php if (!class_exists('WooCommerce')){ echo '<main class="pos"><section class="panel"><h2>Error</h2><div class="toolbar">WooCommerce es requerido.</div></section></main></body></html>'; exit; } ?>
        <?php
        $args = array('status'=>'publish','stock_status'=>'instock','limit'=>-1,'return'=>'objects');
        $prods = wc_get_products($args);
        $items = array();
        foreach ($prods as $p){
            $stock = $p->get_stock_quantity();
            $in = $p->is_in_stock();
            if ($p->managing_stock()){
                if ($stock === null || $stock < 1) continue;
            } else {
                if (!$in) continue;
                $stock = 999999;
            }
            $items[] = array(
                'id'=>$p->get_id(),'title'=>$p->get_name(),'sku'=>$p->get_sku(),
                'price'=>(float)$p->get_price(),'stock'=>(int)$stock,
                'img'=>wp_get_attachment_image_url($p->get_image_id(),'thumbnail'),
                'desc'=>wp_strip_all_tags($p->get_short_description() ?: $p->get_description()),
            );
        } ?>
        <main class="pos">
            <section class="panel left-split">
                <h2>Productos</h2>
                <div class="toolbar"><input id="q" type="search" placeholder="Buscar por título, SKU o descripción (unificado)..."></div>
                <div class="prod-header"><div></div><div>Producto</div><div>SKU</div><div>Precio (€)</div><div>Unidades</div><div></div></div>
                <div class="prod-rows" id="prows"></div>
                <div class="manual-add">
                    <input id="m_title" type="text" placeholder="Título del producto">
                    <input id="m_sku" type="text" placeholder="SKU (opcional)">
                    <input id="m_price" type="text" inputmode="decimal" placeholder="Precio (ej. 14,99)">
                    <input id="m_qty" type="number" min="1" value="1">
                    <button id="m_add" class="btn">Añadir</button>
                </div>
            </section>
            <section class="panel">
                <h2>Carrito</h2>
                <div class="cart-header"><div></div><div>Producto</div><div>SKU</div><div>Precio (€)</div><div>Dto. (%)</div><div>Unidades</div><div>Total</div><div></div></div>
                <div class="cart-rows" id="crows"></div>
                <div id="opciones"><div class="options-body">
                    <button id="btnCobrar" class="btn primary">Cobrar</button>
                    <a href="<?php echo esc_url( add_query_arg(POS_TIENDA_SALES_QUERYVAR,1,home_url('/')) ); ?>" target="_blank" class="btn">Ventas</a>
                    <a href="<?php echo esc_url( add_query_arg(POS_TIENDA_REPORTS_QUERYVAR,1,home_url('/')) ); ?>" target="_blank" class="btn">Informes</a>
                    <a href="<?php echo esc_url( add_query_arg(POS_TIENDA_CLIENTS_QUERYVAR,1,home_url('/')) ); ?>" target="_blank" class="btn">Clientes</a>
                    <a href="<?php echo esc_url( add_query_arg(POS_TIENDA_DAILY_REPORT_QUERYVAR,1,home_url('/')) ); ?>" target="_blank" class="btn">Informe diario</a>
                    <button type="button" id="btnImpuestos" class="btn">Impuestos</button>
<?php
                        $manual_file = (pos_tienda_get_lang() === 'en') ? 'manual/Manual_POS_OfiSoftWeb_EN.pdf' : 'manual/Manual_POS_OfiSoftWeb.pdf';
                        if (!file_exists(plugin_dir_path(__FILE__) . $manual_file)) {
                            $manual_file = 'manual/Manual_POS_OfiSoftWeb.pdf';
                        }
                      ?>
                      <a id="btnInstrucciones" class="btn" target="_blank" href="<?php echo esc_url( plugins_url( $manual_file, __FILE__ ) ); ?>"><?php echo esc_html( pos_tienda_i18n('Instrucciones','Instructions') ); ?></a>
                    <div style="grid-column:1 / -1; display:grid; gap:6px;">
                      <label for="orderNotes" style="font-weight:600">Anotaciones</label>
                      <textarea id="orderNotes" rows="2" placeholder="Observaciones del pedido (se imprimen)" style="width:100%;padding:8px 10px;border:1px solid #d3e9d3;border-radius:10px;resize:vertical"></textarea>
                    </div>
                </div></div>
            </section>
        </main>
        <!-- Modal Cobro -->
        <div class="modal" id="payModal">
            <div class="card">
                <header>Cobro</header>
                <div class="body">
                    <div class="stepper"><div class="dot active" id="dot1"></div><div class="dot" id="dot2"></div></div>
                    <div id="step1">
                        <div class="grid" style="grid-template-columns:2fr 1fr">
                            <label>Método de pago
                                <select id="payMethod"><option value="cash">Efectivo</option><option value="card">Tarjeta</option></select>
                            </label>
                            <label>Total <input id="payTotal" type="text" readonly></label>
                        </div>
                        <div class="grid" id="cashBox" style="grid-template-columns:2fr 1fr">
                            <label>Efectivo recibido <input id="cashGiven" type="text" inputmode="decimal"></label>
                            <label>Cambio <input id="cashChange" type="text" readonly></label>
                        </div>
                    </div>
                    <div id="step2" style="display:none">
                        <div class="grid" style="grid-template-columns:2fr 1fr">
    <label>Tipo de documento
        <select id="docType"><option value="T">Ticket</option><option value="P">Proforma</option><option value="F">Factura</option></select>
    </label>
    <div id="clientSearchBox" style="display:none">
        <label>Cliente (buscar)
            <input id="cliSearch" type="search" placeholder="Nombre, Nº cliente, NIF, empresa, teléfono o email" autocomplete="off">
        </label>
        <div id="cliSearchResults" class="table" style="max-height:140px;overflow:auto;font-size:12px;margin-top:4px;display:none"></div>
    </div>
</div>
                        <div id="invoiceBox" class="grid" style="display:none;grid-template-columns:2fr 1fr">
                            <label>Empresa <input id="invCompany" type="text"></label>
                            <label>Nombre y apellidos <input id="invName" type="text"></label>
                            <label>NIF <input id="invNif" type="text"></label>
                            <label>Dirección <input id="invAddress" type="text"></label>
                            <label>Email <input id="invEmail" type="email"></label>
                            <label>Teléfono <input id="invPhone" type="text"></label>
                        </div>
                    </div>
                </div>
                <div class="footer">
                    <button class="btn" id="payCancel">Cancelar</button>
                    <div><button class="btn" id="payPrev" style="display:none">Anterior</button><button class="btn primary" id="payNext">Siguiente</button><button class="btn primary" id="payCharge" style="display:none">Cobrar</button></div>
                </div>
            </div>
        </div>
<!-- Modal Impuestos -->
        <div class="modal" id="taxModal">
            <div class="card" style="max-width:520px">
                <header>Impuestos</header>
                <div class="body">
                    <div class="grid" style="grid-template-columns:1fr 1fr">
                        <label>Nombre del impuesto
                            <input id="taxName" type="text" placeholder="Impuesto">
                        </label>
                        <label>Porcentaje (%)
                            <input id="taxRate" type="text" inputmode="decimal" placeholder="21">
                        </label>
                    </div>
                    <div style="font-size:12px;opacity:.8">Ejemplo: 21 = 21% (se calcula como 1,21)</div>
                </div>
                <div class="footer">
                    <button class="btn" id="taxCancel">Cancelar</button>
                    <button class="btn primary" id="taxSave">Guardar</button>
                </div>
            </div>
        </div>
        <script>
        (function(){
            function byId(id){ return document.getElementById(id); }
            function fmtEUR(n){ return Number(n||0).toLocaleString(window.POS_LOCALE,{style:'currency',currency:'EUR'}); }
            function round2(n){ n = Number(n||0); return Math.round((n + Number.EPSILON) * 100) / 100; }
            function parseEU(v){ v = String(v||'').replace(/\u00A0/g,' ').trim(); v = v.replace(/\./g,'').replace(',', '.'); var n = parseFloat(v); return isNaN(n)?0:n; }
            function parsePct(v){ v = String(v||'').replace(/\u00A0/g,' ').trim(); v = v.replace(/%/g,'').replace(/\s+/g,''); if (v.indexOf(',') !== -1){ v = v.replace(/\./g,''); v = v.replace(',', '.'); } var n = parseFloat(v); return isNaN(n)?0:n; }
            var DATA = <?php echo wp_json_encode($items); ?>;
            var POS_EDIT_ORDER_ID = <?php echo isset($_GET['order_id']) ? (int) $_GET['order_id'] : 0; ?>;
            var POS_TAX = { name: <?php echo wp_json_encode( (string) get_option('pos_tienda_tax_name', 'Impuesto') ); ?>, rate: <?php echo wp_json_encode( (float) get_option('pos_tienda_tax_rate', 0 ) ); ?> };
            var prows = byId('prows'), crows = byId('crows'), q = byId('q');
            var cart = [];
            function setCartFromOrder(items){
                cart = [];
                (items || []).forEach(function(it){
                    var obj = {
                        id: it.id,
                        title: it.title || it.name || '',
                        sku: it.sku || '',
                        price: it.price || 0,
                        qty: it.qty || 1,
                        disc: it.disc || 0
                    };
                    if (typeof it.stock === 'number' && it.stock > 0){
                        obj.stock = it.stock;
                    }
                    cart.push(obj);
                });
                renderCart();
            }
            function loadOrderForEdit(orderId){
                if (!orderId) return;
                var fd = new FormData();
                fd.append('action','pos_tienda_load_order');
                fd.append('order_id', orderId);
                fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>', {
                    method:'POST',
                    credentials:'same-origin',
                    body:fd
                }).then(function(r){ return r.json(); }).then(function(d){
                    if (!d || !d.success || !d.data){ return; }
                    var data = d.data;
                    if (data.items){ setCartFromOrder(data.items); }
                    if (data.doc && byId('docType')){
                        byId('docType').value = data.doc;
                        if (typeof updateInvoiceBox === 'function'){ updateInvoiceBox(); }
                    }
                    if (data.billing){
                        var b = data.billing;
                        if (byId('invCompany')) byId('invCompany').value = b.company || '';
                        if (byId('invName'))    byId('invName').value    = b.name || '';
                        if (byId('invNif'))     byId('invNif').value     = b.nif || '';
                        if (byId('invAddress')) byId('invAddress').value = b.address || '';
                        if (byId('invPhone'))   byId('invPhone').value   = b.phone || '';
                        if (byId('invEmail'))   byId('invEmail').value   = b.email || '';
                    }
                    if (data.notes && byId('orderNotes')){
                        byId('orderNotes').value = data.notes;
                    }
                }).catch(function(err){
                    if (window.console && console.error){ console.error(err); }
                });
            }
            function renderProducts(list){
                prows.innerHTML='';
                (list||[]).forEach(function(p){
                    var row = document.createElement('div'); row.className='prod-row';
                    row.innerHTML = '<div>'+ (p.img?('<img src="'+p.img+'" style="width:56px;height:56px;object-fit:cover;border-radius:8px">'):'') +'</div>'
                                   + '<div class="title">'+p.title+'</div>'
                                   + '<div>'+(p.sku||'')+'</div>'
                                   + '<div class="price" data-price="'+p.price+'">'+fmtEUR(p.price)+'</div>'
                                   + '<div><input class="qty" type="number" min="1" max="'+p.stock+'" value="1"></div>'
                                   + '<div><button class="btn add">Añadir</button></div>';
                    row.querySelector('.add').addEventListener('click', function(){
                        var qty = parseInt(row.querySelector('.qty').value||'1',10);
                        if (qty<1) qty=1; if (qty>p.stock){ alert('No hay stock suficiente'); return; }
                        addToCart({id:p.id, title:p.title, sku:p.sku, price:p.price, qty:qty, stock:p.stock});
                    });
                    prows.appendChild(row);
                });
            }
            byId('m_add').addEventListener('click', function(){
                var title=byId('m_title').value.trim(); if(!title){ alert('Título requerido'); return; }
                var sku=byId('m_sku').value.trim();
                var price=parseEU(byId('m_price').value); if(price<=0){ alert('Precio inválido'); return; }
                var qty=parseInt(byId('m_qty').value||'1',10); if(qty<1)qty=1;
                addToCart({id:0,title:title,sku:sku,price:price,qty:qty});
                byId('m_title').value=''; byId('m_sku').value=''; byId('m_price').value=''; byId('m_qty').value=1;
            });
            function addToCart(item){
                // Buscar línea existente en el carrito (mismo producto, precio y SKU)
                var ex = cart.find(function(x){
                    return x.id === item.id && x.price === item.price && (x.sku || '') === (item.sku || '');
                });
                // Stock máximo permitido (solo si viene definido)
                var maxStock = (typeof item.stock === 'number' && item.stock > 0) ? item.stock : null;
                if (ex) {
                    // Preferir el stock guardado en el propio item del carrito si existe
                    var stock = (typeof ex.stock === 'number' && ex.stock > 0) ? ex.stock : maxStock;
                    if (stock !== null && (ex.qty + item.qty) > stock) {
                        alert('No hay stock suficiente');
                        return;
                    }
                    ex.qty += item.qty;
                } else {
                    var obj = {
                        id: item.id,
                        title: item.title,
                        sku: item.sku,
                        price: item.price,
                        qty: item.qty,
                        disc: 0
                    };
                    if (maxStock !== null) {
                        obj.stock = maxStock;
                    }
                    cart.push(obj);
                }
                renderCart();
            }

function totalCart(){ var t=0; cart.forEach(function(c){ t += c.price*c.qty*(1-(c.disc||0)/100); }); return t; }
            function renderCart(){
                crows.innerHTML=''; var total=0;
                cart.forEach(function(c,idx){
                    var priceText = (Number(c.price)||0).toLocaleString(window.POS_LOCALE,{minimumFractionDigits:2, maximumFractionDigits:2});
                    var line = c.price*c.qty*(1-(c.disc||0)/100); total+=line;
                    var row = document.createElement('div'); row.className='cart-row';
                    row.innerHTML = '<div></div><div>'+c.title+'</div><div>'+(c.sku||'')+'</div>'
                                  + '<div><input class="price" type="text" inputmode="decimal" value="'+priceText+'"></div>'
                                  + '<div><input class="disc" type="number" min="0" max="100" value="'+(c.disc||0)+'"></div>'
                                  + '<div><input class="qty" type="number" min="1" value="'+c.qty+'"></div>'
                                  + '<div class="total">'+fmtEUR(line)+'</div>'
                                  + '<div><button class="btn del">✕</button></div>';
                    var pInput = row.querySelector('.price');
                    var dInput = row.querySelector('.disc');
                    var qInput = row.querySelector('.qty');
                    pInput.addEventListener('change', function(e){
                        var v = parseEU(e.target.value||'0'); if (v<0) v=0;
                        c.price = v; renderCart();
                    });
                    dInput.addEventListener('change', function(e){
                        c.disc=parseFloat(e.target.value||'0'); renderCart();
                    });
                    qInput.addEventListener('change', function(e){
                        var v = parseInt(e.target.value||'1',10);
                        if (v < 1) v = 1;
                        if (typeof c.stock === 'number' && c.stock > 0 && v > c.stock) {
                            alert('No hay stock suficiente');
                            v = c.stock;
                        }
                        c.qty = v;
                        renderCart();
                    });
                    row.querySelector('.del').addEventListener('click', function(){ cart.splice(idx,1); renderCart(); });
                    crows.appendChild(row);
                });
                var totalIncl = round2(total);
                var rate = (POS_TAX && typeof POS_TAX.rate !== 'undefined') ? parseFloat(POS_TAX.rate) : 0;
                if (!isFinite(rate) || rate < 0) rate = 0;
                var totalSin = totalIncl;
                var taxAmt = 0;
                if (rate > 0) {
                    var div = 1 + (rate / 100);
                    if (div !== 0) {
                        totalSin = totalIncl / div;
                        taxAmt = totalIncl - totalSin;
                    }
                }
                totalSin = round2(totalSin);
                taxAmt   = round2(taxAmt);
                var f=document.createElement('div');
                f.style.cssText='padding:8px 12px;text-align:right;font-weight:700;display:grid;gap:2px;';
                var tname = (POS_TAX && POS_TAX.name) ? String(POS_TAX.name) : 'Impuesto';
                tname = tname.trim() || 'Impuesto';
                var l1=document.createElement('div'); l1.style.fontWeight='600'; l1.textContent='Total sin '+tname+': '+fmtEUR(totalSin);
                var l2=document.createElement('div'); l2.style.fontWeight='600'; l2.textContent=tname+': '+fmtEUR(taxAmt);
                var l3=document.createElement('div'); l3.textContent='Total '+tname+' incuido: '+fmtEUR(totalIncl);
                f.appendChild(l1); f.appendChild(l2); f.appendChild(l3);
                crows.appendChild(f);
            }
            q.addEventListener('input', function(){
                var s=q.value.trim().toLowerCase(); if(!s){ renderProducts(DATA); return; }
                var out=DATA.filter(function(p){
                    return (p.title||'').toLowerCase().includes(s) || (p.sku||'').toLowerCase().includes(s) || (p.desc||'').toLowerCase().includes(s);
                }); renderProducts(out);
            }); renderProducts(DATA);
            if (typeof POS_EDIT_ORDER_ID !== 'undefined' && POS_EDIT_ORDER_ID > 0) {
                loadOrderForEdit(POS_EDIT_ORDER_ID);
            }

            // Pago
            var payModal=byId('payModal'), dot1=byId('dot1'),dot2=byId('dot2'), step1=byId('step1'),step2=byId('step2');
            var btnNext=byId('payNext'),btnPrev=byId('payPrev'),btnCharge=byId('payCharge'),btnCancel=byId('payCancel');
            var payMethod=byId('payMethod'),cashBox=byId('cashBox'),payTotal=byId('payTotal'),cashGiven=byId('cashGiven'),cashChange=byId('cashChange');
            var docSelect=byId('docType');
            function openPay(){ if(!cart.length){alert('Carrito vacío');return;} payTotal.value=fmtEUR(totalCart()); cashGiven.value=''; cashChange.value=fmtEUR(0); payMethod.value='cash'; cashBox.style.display='grid'; docSelect.value='T'; updateInvoiceBox(); step1.style.display='block'; step2.style.display='none'; btnNext.style.display='inline-flex'; btnCharge.style.display='none'; btnPrev.style.display='none'; dot1.classList.add('active'); dot2.classList.remove('active'); payModal.style.display='flex'; }
            function closePay(){ payModal.style.display='none'; }
            document.getElementById('btnCobrar').addEventListener('click', openPay); btnCancel.addEventListener('click', closePay);
            payMethod.addEventListener('change', function(){ cashBox.style.display=(payMethod.value==='cash')?'grid':'none'; });
            cashGiven.addEventListener('input', function(){ var tg=totalCart(), cg=parseEU(cashGiven.value||'0'); var ch=(cg-tg); cashChange.value=fmtEUR(ch); });
            function docType(){ return docSelect.value||'T'; }
            function updateInvoiceBox(){
                var show = (docType()==='F' || docType()==='P');
                var box  = document.getElementById('invoiceBox');
                var cliBox = document.getElementById('clientSearchBox');
                if (box)    box.style.display    = show ? 'grid' : 'none';
                if (cliBox) cliBox.style.display = show ? 'block' : 'none';
            }
            docSelect.addEventListener('change', updateInvoiceBox);
            var cliSearch   = document.getElementById('cliSearch');
            var cliResults  = document.getElementById('cliSearchResults');
            var cliTimer    = null;

            function clearClientResults(){
                if (cliResults){
                    cliResults.innerHTML = '';
                    cliResults.style.display = 'none';
                }
            }
            function applyClientToInvoice(c){
                if (!c) return;
                var el;
                el = document.getElementById('invCompany'); if (el) el.value = c.company || '';
                el = document.getElementById('invName');    if (el) el.value = c.name || '';
                el = document.getElementById('invNif');     if (el) el.value = c.nif || '';
                el = document.getElementById('invAddress'); if (el) el.value = c.address || '';
                el = document.getElementById('invPhone');   if (el) el.value = c.phone || '';
                el = document.getElementById('invEmail');   if (el) el.value = c.email || '';
            }
            function renderClientResults(list){
                if (!cliResults) return;
                cliResults.innerHTML = '';
                if (!list || !list.length){
                    cliResults.style.display = 'none';
                    return;
                }
                list.forEach(function(c){
                    var row = document.createElement('div');
                    row.className = 'row';
                    row.style.cursor = 'pointer';
                    row.style.padding = '4px 6px';
                    row.style.borderBottom = '1px solid #d3e9d3';
                    row.textContent = (c.id ? c.id + ' - ' : '') + (c.name || '') + (c.company ? ' ('+c.company+')' : '') + (c.nif ? ' ['+c.nif+']' : '');
                    row.addEventListener('click', function(){
                        applyClientToInvoice(c);
                        clearClientResults();
                    });
                    cliResults.appendChild(row);
                });
                cliResults.style.display = 'block';
            }
            function searchClients(term){
                if (!term || term.length < 2) { clearClientResults(); return; }
                var fd = new FormData();
                fd.append('action','pos_tienda_search_clients');
                fd.append('q', term);
                fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>', {
                    method:'POST',
                    credentials:'same-origin',
                    body:fd
                }).then(function(r){ return r.json(); }).then(function(d){
                    if (d && d.success && d.data && d.data.items){
                        renderClientResults(d.data.items);
                    } else {
                        clearClientResults();
                    }
                }).catch(function(){
                    clearClientResults();
                });
            }
            if (cliSearch){
                cliSearch.addEventListener('input', function(){
                    var v = (this.value || '').trim();
                    if (cliTimer) clearTimeout(cliTimer);
                    if (!v || v.length < 2){
                        clearClientResults();
                        return;
                    }
                    cliTimer = setTimeout(function(){ searchClients(v); }, 300);
                });
            }

            btnNext.addEventListener('click', function(){ if(payMethod.value==='cash'){var tg=totalCart(),cg=parseEU(cashGiven.value||'0'); if(isNaN(cg)||cg<tg){alert('Efectivo insuficiente');return;}} step1.style.display='none'; step2.style.display='block'; btnNext.style.display='none'; btnCharge.style.display='inline-flex'; btnPrev.style.display='inline-flex'; dot1.classList.remove('active'); dot2.classList.add('active'); updateInvoiceBox(); });
            btnPrev.addEventListener('click', function(){ step2.style.display='none'; step1.style.display='block'; btnNext.style.display='inline-flex'; btnCharge.style.display='none'; btnPrev.style.display='none'; dot2.classList.remove('active'); dot1.classList.add('active'); });
            btnCharge.addEventListener('click', function(){
                if (docType()==='F' || docType()==='P'){ var invName=byId('invName').value.trim(), invNif=byId('invNif').value.trim(), invAddress=byId('invAddress').value.trim(), invEmail=byId('invEmail').value.trim(); if(!invName||!invNif||!invAddress||!invEmail){ alert('Completa los datos del receptor'); return; } }
                var fd=new FormData(); fd.append('cart', JSON.stringify(cart));
                fd.append('action','pos_tienda_make_order');
                fd.append('order_id', (typeof POS_EDIT_ORDER_ID !== 'undefined' && POS_EDIT_ORDER_ID > 0) ? POS_EDIT_ORDER_ID : 0);
                fd.append('doc', byId('docType')?byId('docType').value:'T');
                fd.append('pay_method', (payMethod && payMethod.value) ? payMethod.value : 'cash');
                if (byId('docType') && (byId('docType').value==='F' || byId('docType').value==='P')){
                  fd.append('inv_company', (byId('invCompany')||{}).value||'');
                  fd.append('inv_name',    (byId('invName')||{}).value||'');
                  fd.append('inv_nif',     (byId('invNif')||{}).value||'');
                  fd.append('inv_address', (byId('invAddress')||{}).value||'');
                  fd.append('inv_email',   (byId('invEmail')||{}).value||'');
                  fd.append('inv_phone',   (byId('invPhone')||{}).value||'');
                }
                fd.append('cash_given',cashGiven.value||'');
                fd.append('notes', (byId('orderNotes') && byId('orderNotes').value) ? byId('orderNotes').value.trim() : '');
                var printWin=window.open('', '_blank');
                fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>', { method:'POST', credentials:'same-origin', body:fd }).then(r=>r.json()).then(function(d){
                    if(!d||!d.success){ alert((d&&d.data&&d.data.message)?d.data.message:'Error al cobrar'); try{printWin.close();}catch(e){} return; }
                    try{ printWin.location.href=d.data.receipt_url; }catch(e){ window.open(d.data.receipt_url,'_blank'); }
                    cart=[]; renderCart(); closePay();
                    try{ var n=byId('orderNotes'); if(n) n.value=''; }catch(e){}
                    // Recargar la página automáticamente tras completar la venta
                    setTimeout(function(){ window.location.reload(); }, 800);
                }).catch(function(){ alert('Error de conexión'); try{printWin.close();}catch(e){} });
            });

            var btnIssuer = byId('btnIssuer');
            if (btnIssuer) {
              btnIssuer.addEventListener('click', function(){
                var modal = byId('issuerModal');
                if (!modal) return;
                var fd = new FormData(); fd.append('action','pos_tienda_get_settings');
                fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>',{method:'POST',credentials:'same-origin',body:fd})
                  .then(r=>r.json()).then(function(d){
                    var s  = d && d.success && d.data ? d.data.issuer : {};
                    var st = d && d.success && d.data ? (d.data.issuer_tickets || {}) : {};
                    // Facturas / proformas
                    (byId('emiCompany')||{}).value = (s&&s.company)||'OfiSoftWeb';
                    (byId('emiName')||{}).value    = (s&&s.name)||'';
                    (byId('emiNif')||{}).value     = (s&&s.nif)||'';
                    (byId('emiAddress')||{}).value = (s&&s.address)||'';
                    (byId('emiEmail')||{}).value   = (s&&s.email)||'tienda@OfiSoftWeb.com';
                    (byId('emiPhone')||{}).value   = (s&&s.phone)||'';
                    // Tiques
                    (byId('emiTCompany')||{}).value = (st&&st.company)||'OfiSoftWeb';
                    (byId('emiTName')||{}).value    = (st&&st.name)||'';
                    (byId('emiTNif')||{}).value     = (st&&st.nif)||'';
                    (byId('emiTAddress')||{}).value = (st&&st.address)||'';
                    (byId('emiTEmail')||{}).value   = (st&&st.email)||'tienda@OfiSoftWeb.com';
                    (byId('emiTPhone')||{}).value   = (st&&st.phone)||'';
                    modal.style.display='flex';
                  }).catch(function(){ modal.style.display='flex'; });
              });
            }

            // Impuestos
            var btnImpuestos = byId('btnImpuestos');
            var taxModal = byId('taxModal');

            function fillTaxInputs(){
                (byId('taxName')||{}).value = (POS_TAX && POS_TAX.name) ? POS_TAX.name : '';
                var r = (POS_TAX && (POS_TAX.rate || POS_TAX.rate===0)) ? POS_TAX.rate : '';
                try{ r = String(r).replace('.', ','); }catch(e){}
                (byId('taxRate')||{}).value = r;
            }

            function openTaxModal(){
                if (!taxModal) return;
                var fd = new FormData(); fd.append('action','pos_tienda_get_settings');
                fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>',{method:'POST',credentials:'same-origin',body:fd})
                  .then(r=>r.json()).then(function(d){
                    if (d && d.success && d.data){
                        if (typeof d.data.tax_name !== 'undefined') POS_TAX.name = d.data.tax_name;
                        if (typeof d.data.tax_rate !== 'undefined') POS_TAX.rate = d.data.tax_rate;
                    }
                    fillTaxInputs();
                    taxModal.style.display='flex';
                  }).catch(function(){
                    fillTaxInputs();
                    taxModal.style.display='flex';
                  });
            }

            if (btnImpuestos) {
                btnImpuestos.addEventListener('click', openTaxModal);
            }

            var taxCancel = byId('taxCancel');
            if (taxCancel && taxModal){
                taxCancel.addEventListener('click', function(){ taxModal.style.display='none'; });
            }

            var taxSave = byId('taxSave');
            if (taxSave && taxModal){
                taxSave.addEventListener('click', function(){
                    var name = ((byId('taxName')||{}).value||'').trim();
                    var rate = parsePct((byId('taxRate')||{}).value||'');
                    var fd = new FormData();
                    fd.append('action','pos_tienda_save_tax');
                    fd.append('tax_name', name);
                    fd.append('tax_rate', rate);
                    fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>',{method:'POST',credentials:'same-origin',body:fd})
                      .then(r=>r.json()).then(function(d){
                        if (d && d.success && d.data){
                            POS_TAX.name = d.data.tax_name || name || 'Impuesto';
                            POS_TAX.rate = (typeof d.data.tax_rate !== 'undefined') ? d.data.tax_rate : rate;
                            alert('Impuestos guardados');
                            taxModal.style.display='none';
                            renderCart();
                        } else {
                            alert((d&&d.data&&d.data.message)||'No se pudo guardar');
                        }
                      }).catch(function(){ alert('Error de conexión'); });
                });
            }
        })();
        </script>
    <?php elseif ($is_reports): ?>
        <main class="pos"><section class="panel"><h2>Informes POS</h2>
            <div class="toolbar" style="flex-wrap:wrap">
                <label>Desde&nbsp;<input id="r_from" type="date"></label>
                <label>Hasta&nbsp;<input id="r_to" type="date"></label>
                <label>Método&nbsp;
                    <select id="r_method">
                        <option value="all">Todos</option>
                        <option value="cash">Efectivo</option>
                        <option value="card">Tarjeta</option>
                    </select>
                </label>
                <button id="r_go" class="btn">Buscar</button>
                <div style="margin-left:auto;display:flex;gap:8px">
                    <button id="r_print" class="btn">Imprimir</button>
                    <button id="r_home" class="btn">Inicio</button>
                </div>
                <div style="width:100%;font-size:12px;color:#000;opacity:.8">Nota: Pulsa "Inicio" para volver al POS.</div>
            </div>
            <div id="r_summary" style="padding:12px;display:grid;grid-template-columns:repeat(4,1fr);gap:12px"></div>
            <div class="body" style="padding:12px;overflow:auto;max-height:calc(100vh - 220px)" id="r_area">
                <table class="table" id="r_table">
                    <thead><tr><th>Código</th><th>Fecha</th><th>Método</th><th>Total</th><th>Estado</th></tr></thead>
                    <tbody></tbody>
                </table>
            </div>
        </section></main>
        <script>
        (function(){
            function byId(i){return document.getElementById(i);} function eur(n){return Number(n||0).toLocaleString(window.POS_LOCALE,{style:'currency',currency:'EUR'});} function ymd(d){return d.toISOString().slice(0,10);} var today=new Date(); var from=new Date(); from.setDate(today.getDate()-7); byId('r_from').value=ymd(from); byId('r_to').value=ymd(today);
            function render(data){ var sum=byId('r_summary'); var tCash=data.total_cash||0, tCard=data.total_card||0, tAll=(data.total||0), cnt=(data.count||0); sum.innerHTML=''
                +'<div class="panel" style="padding:12px"><div>Total POS</div><div style="font-weight:700">'+eur(tAll)+'</div></div>'
                +'<div class="panel" style="padding:12px"><div>Efectivo</div><div style="font-weight:700">'+eur(tCash)+'</div></div>'
                +'<div class="panel" style="padding:12px"><div>Tarjeta</div><div style="font-weight:700">'+eur(tCard)+'</div></div>'
                +'<div class="panel" style="padding:12px"><div>Pedidos</div><div style="font-weight:700">'+cnt+'</div></div>';
                var tb=byId('r_table').querySelector('tbody'); tb.innerHTML=''; (data.rows||[]).forEach(function(r){ var tr=document.createElement('tr'); tr.innerHTML='<td>'+r.code+'</td><td>'+r.date+'</td><td>'+r.method+'</td><td>'+eur(r.total)+'</td><td>'+r.status+'</td>'; tb.appendChild(tr); });
            }
            function load(){ var fd=new FormData(); fd.append('action','pos_tienda_reports'); fd.append('from', byId('r_from').value); fd.append('to', byId('r_to').value); fd.append('method', byId('r_method').value); fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>',{method:'POST',credentials:'same-origin',body:fd}).then(r=>r.json()).then(function(d){ if(d&&d.success){ render(d.data);} else { alert('No se pudo cargar'); } }).catch(function(){ alert('Error de conexión'); }); }
            byId('r_go').addEventListener('click', load); byId('r_print').addEventListener('click', function(){ window.print(); }); byId('r_home').addEventListener('click', function(){ location.href='<?php echo esc_js( esc_url( add_query_arg(POS_TIENDA_QUERYVAR,1,home_url('/')) ) ); ?>'; });
            load();
        })();
        </script>
        <?php elseif ($is_clients): ?>
        <main class="pos"><section class="panel">
            <h2 style="display:flex;align-items:center;justify-content:space-between">
                Clientes POS
                <span>
                    <a class="btn" href="<?php echo esc_url( add_query_arg( POS_TIENDA_QUERYVAR, 1, home_url( '/' ) ) ); ?>">Inicio</a>
                    <button type="button" class="btn primary" id="btnClientsManage">Nuevo cliente</button>
                </span>
            </h2>
            <div class="body" style="padding:12px;overflow:auto;max-height:calc(100vh - 140px)">
                <div style="margin-bottom:8px;display:flex;justify-content:flex-end">
                    <input type="search" id="clientsSearch" placeholder="Buscar cliente (nombre, empresa, NIF, teléfono, email)" style="max-width:260px;width:100%;padding:4px 8px;">
                </div>
                <table class="table" id="clientsTable">
                    <thead>
                        <tr>
                            <th style="width:60px">Nº</th>
                            <th>Nombre</th>
                            <th>Empresa</th>
                            <th>NIF</th>
                            <th>Dirección</th>
                            <th>Teléfono</th>
                            <th>Email</th>
                            <th style="width:120px">Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $clients = get_option('pos_tienda_clients', array());
                        if (!is_array($clients)) { $clients = array(); }
                        usort($clients, function($a, $b){
                            $ia = isset($a['id']) ? (int) $a['id'] : 0;
                            $ib = isset($b['id']) ? (int) $b['id'] : 0;
                            if ($ia === $ib) return 0;
                            return ($ia < $ib) ? -1 : 1;
                        });
                        if (empty($clients)){
                            echo '<tr><td colspan="8" style="padding:8px;color:#666;font-size:13px">Todavía no hay clientes guardados.</td></tr>';
                        } else {
                            foreach ($clients as $c){
                                if (!is_array($c)) continue;
                                $id      = isset($c['id'])      ? (int) $c['id']      : 0;
                                $name    = isset($c['name'])    ? $c['name']         : '';
                                $company = isset($c['company']) ? $c['company']      : '';
                                $nif     = isset($c['nif'])     ? $c['nif']          : '';
                                $address = isset($c['address']) ? $c['address']      : '';
                                $phone   = isset($c['phone'])   ? $c['phone']        : '';
                                $email   = isset($c['email'])   ? $c['email']        : '';
                                $notes   = isset($c['notes'])   ? $c['notes']        : '';
                                echo '<tr class="cli-row"'
                                    . ' data-id="'.esc_attr($id).'"'
                                    . ' data-name="'.esc_attr($name).'"'
                                    . ' data-company="'.esc_attr($company).'"'
                                    . ' data-nif="'.esc_attr($nif).'"'
                                    . ' data-address="'.esc_attr($address).'"'
                                    . ' data-phone="'.esc_attr($phone).'"'
                                    . ' data-email="'.esc_attr($email).'"'
                                    . ' data-notes="'.esc_attr($notes).'"'
                                    . '>'
                                    . '<td>'.esc_html($id).'</td>'
                                    . '<td>'.esc_html($name).'</td>'
                                    . '<td>'.esc_html($company).'</td>'
                                    . '<td>'.esc_html($nif).'</td>'
                                    . '<td>'.esc_html($address).'</td>'
                                    . '<td>'.esc_html($phone).'</td>'
                                    . '<td>'.esc_html($email).'</td>'
                                    . '<td>'
                                        . '<button type="button" class="btn" style="font-size:11px;padding:2px 6px;margin-right:4px" data-act="edit">Editar</button>'
                                        . '<button type="button" class="btn bad" style="font-size:11px;padding:2px 6px" data-act="del">Eliminar</button>'
                                    . '</td>'
                                    . '</tr>';
                            }
                        }
                        ?>
                    </tbody>
                </table>
            </div>
        </section></main>

        <!-- Modal de cliente (crear / editar) -->
        <div class="modal" id="clientsModal">
            <div class="card" style="max-width:640px;width:100%;">
                <header>Cliente POS</header>
                <div class="body">
                    <div class="grid" style="grid-template-columns:1fr 1fr">
                        <label>Nº cliente <input id="cliId" type="text" readonly placeholder="(nuevo)"></label>
                        <label>NIF <input id="cliNif" type="text"></label>
                        <label>Nombre y apellidos <input id="cliName" type="text"></label>
                        <label>Empresa <input id="cliCompany" type="text"></label>
                        <label>Dirección <input id="cliAddress" type="text"></label>
                        <label>Teléfono <input id="cliPhone" type="text"></label>
                        <label>Email <input id="cliEmail" type="email"></label>
                    </div>
                    <label>Notas <textarea id="cliNotes" style="width:100%;min-height:60px"></textarea></label>
                </div>
                <div class="footer" style="display:flex;justify-content:space-between;gap:8px;flex-wrap:wrap">
                    <div style="display:flex;gap:8px;flex-wrap:wrap">
                        <button class="btn" id="cliNew">Nuevo</button>
                        <button class="btn bad" id="cliDelete">Eliminar</button>
                    </div>
                    <div style="display:flex;gap:8px;flex-wrap:wrap">
                        <button class="btn" id="cliCancel">Cerrar</button>
                        <button class="btn primary" id="cliSave">Guardar cliente</button>
                    </div>
                </div>
            </div>
        </div>

        <script>
        (function(){
            function byId(id){ return document.getElementById(id); }
            var cliModal  = byId('clientsModal');
            var btnManage = byId('btnClientsManage');
            var cliCancel = byId('cliCancel');
            var cliSave   = byId('cliSave');
            var cliNew    = byId('cliNew');
            var cliDelete = byId('cliDelete');
            var clientsTable = byId('clientsTable');
            var clientsSearch = byId('clientsSearch');

            if (clientsSearch && clientsTable) {
                clientsSearch.addEventListener('input', function(){
                    var term = (clientsSearch.value || '').toLowerCase().trim();
                    var rows = clientsTable.querySelectorAll('tbody tr');
                    rows.forEach(function(row){
                        if (!term) { row.style.display = ''; return; }
                        var txt = row.textContent || '';
                        txt = txt.toLowerCase();
                        row.style.display = txt.indexOf(term) !== -1 ? '' : 'none';
                    });
                });
            }

            function clearClientForm(){
                if (byId('cliId'))      byId('cliId').value      = '';
                if (byId('cliNif'))     byId('cliNif').value     = '';
                if (byId('cliName'))    byId('cliName').value    = '';
                if (byId('cliCompany')) byId('cliCompany').value = '';
                if (byId('cliAddress')) byId('cliAddress').value = '';
                if (byId('cliPhone'))   byId('cliPhone').value   = '';
                if (byId('cliEmail'))   byId('cliEmail').value   = '';
                if (byId('cliNotes'))   byId('cliNotes').value   = '';
            }
            function fillClientFormFromRow(row){
                if (!row || !row.dataset) return;
                if (byId('cliId'))      byId('cliId').value      = row.dataset.id || '';
                if (byId('cliNif'))     byId('cliNif').value     = row.dataset.nif || '';
                if (byId('cliName'))    byId('cliName').value    = row.dataset.name || '';
                if (byId('cliCompany')) byId('cliCompany').value = row.dataset.company || '';
                if (byId('cliAddress')) byId('cliAddress').value = row.dataset.address || '';
                if (byId('cliPhone'))   byId('cliPhone').value   = row.dataset.phone || '';
                if (byId('cliEmail'))   byId('cliEmail').value   = row.dataset.email || '';
                if (byId('cliNotes'))   byId('cliNotes').value   = row.dataset.notes || '';
            }
            function openModal(){
                if (!cliModal) return;
                cliModal.style.display = 'flex';
            }
            function closeModal(){
                if (!cliModal) return;
                cliModal.style.display = 'none';
            }

            if (btnManage){
                btnManage.addEventListener('click', function(e){
                    e.preventDefault();
                    clearClientForm();
                    openModal();
                });
            }
            if (cliCancel){
                cliCancel.addEventListener('click', function(e){
                    e.preventDefault();
                    closeModal();
                });
            }
            if (cliNew){
                cliNew.addEventListener('click', function(e){
                    e.preventDefault();
                    clearClientForm();
                });
            }

            if (cliDelete){
                cliDelete.addEventListener('click', function(e){
                    e.preventDefault();
                    var idVal = byId('cliId') ? (byId('cliId').value||'').trim() : '';
                    if (!idVal){
                        alert('Selecciona primero un cliente.');
                        return;
                    }
                    if (!window.confirm('¿Eliminar este cliente?')) return;
                    var fd = new FormData();
                    fd.append('action','pos_tienda_delete_client');
                    fd.append('id', idVal);
                    fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>', {
                        method:'POST',
                        credentials:'same-origin',
                        body:fd
                    }).then(function(r){ return r.json(); }).then(function(d){
                        if (d && d.success){
                            window.location.reload();
                        } else {
                            alert((d && d.data && d.data.message) || 'No se pudo eliminar el cliente');
                        }
                    }).catch(function(){
                        alert('Error de conexión');
                    });
                });
            }

            if (cliSave){
                cliSave.addEventListener('click', function(e){
                    e.preventDefault();
                    var fd = new FormData();
                    fd.append('action','pos_tienda_save_client');
                    fd.append('id', (byId('cliId')||{}).value||'');
                    fd.append('name', (byId('cliName')||{}).value||'');
                    fd.append('company', (byId('cliCompany')||{}).value||'');
                    fd.append('nif', (byId('cliNif')||{}).value||'');
                    fd.append('address', (byId('cliAddress')||{}).value||'');
                    fd.append('phone', (byId('cliPhone')||{}).value||'');
                    fd.append('email', (byId('cliEmail')||{}).value||'');
                    fd.append('notes', (byId('cliNotes')||{}).value||'');
                    fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>', {
                        method:'POST',
                        credentials:'same-origin',
                        body:fd
                    }).then(function(r){ return r.json(); }).then(function(d){
                        if (d && d.success && d.data){
                            alert('Cliente guardado');
                            window.location.reload();
                        } else {
                            alert((d && d.data && d.data.message) || 'No se pudo guardar el cliente');
                        }
                    }).catch(function(){
                        alert('Error de conexión');
                    });
                });
            }

            var rows = document.querySelectorAll('#clientsTable tbody tr.cli-row');
            if (rows && rows.length){
                rows.forEach(function(row){
                    row.addEventListener('click', function(){
                        clearClientForm();
                        fillClientFormFromRow(row);
                        openModal();
                    });
                    var be = row.querySelector('button[data-act="edit"]');
                    if (be){
                        be.addEventListener('click', function(ev){
                            ev.preventDefault();
                            ev.stopPropagation();
                            clearClientForm();
                            fillClientFormFromRow(row);
                            openModal();
                        });
                    }
                    var bd = row.querySelector('button[data-act="del"]');
                    if (bd){
                        bd.addEventListener('click', function(ev){
                            ev.preventDefault();
                            ev.stopPropagation();
                            var idVal = row.dataset.id || '';
                            if (!idVal){
                                alert('ID de cliente no válido.');
                                return;
                            }
                            if (!window.confirm('¿Eliminar este cliente?')) return;
                            var fd = new FormData();
                            fd.append('action','pos_tienda_delete_client');
                            fd.append('id', idVal);
                            fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>', {
                                method:'POST',
                                credentials:'same-origin',
                                body:fd
                            }).then(function(r){ return r.json(); }).then(function(d){
                                if (d && d.success){
                                    window.location.reload();
                                } else {
                                    alert((d && d.data && d.data.message) || 'No se pudo eliminar el cliente');
                                }
                            }).catch(function(){
                                alert('Error de conexión');
                            });
                        });
                    }
                });
            }
        })();
        </script>

    <?php else: ?>

        <main class="pos"><section class="panel">
            <h2 style="display:flex;align-items:center;justify-content:space-between">
                Ventas POS
                <a class="btn" href="<?php echo esc_url( add_query_arg(POS_TIENDA_QUERYVAR,1,home_url('/')) ); ?>">Inicio</a>
            </h2>
            <div class="body" style="padding:12px;overflow:auto;max-height:calc(100vh - 140px)">
                <div style="margin-bottom:8px;display:flex;gap:8px;align-items:center;justify-content:flex-end;flex-wrap:wrap">
                    <select id="salesFilter" style="max-width:200px;width:100%;padding:4px 8px;">
                        <option value="all">Todas</option>
                        <option value="T">Tickets</option>
                        <option value="P">Proformas</option>
                        <option value="F">Facturas</option>
                    </select>
                    <input type="search" id="salesSearch" placeholder="Buscar venta (código, documento, fecha, total, estado)" style="max-width:260px;width:100%;padding:4px 8px;">
                </div>
                <table class="table" id="salesTable">
                    <thead><tr><th>Código</th><th>Fecha</th><th>Total</th><th>Estado</th><th>Acciones</th></tr></thead>
                    <tbody>
                    <?php
                    $args = array('type'=>'shop_order','limit'=>-1,'orderby'=>'date','order'=>'DESC','status'=>'any','return'=>'objects','meta_query'=>array('relation'=>'OR',array('key'=>'_pos_origin','value'=>'pos_tienda','compare'=>'='),array('key'=>'_pos_doc_code','compare'=>'EXISTS')));
                    try { $q=new WC_Order_Query($args); $orders=$q->get_orders(); } catch (Throwable $e){ $orders=array(); }
                    if ($orders){
                        foreach ($orders as $order){
                            $oid=$order->get_id(); $code=$order->get_meta('_pos_doc_code',true) ?: $oid;
                            $doc_type = $order->get_meta('_pos_doc_type', true);
                            if (!$doc_type){
                                $c = is_string($code) ? $code : '';
                                if (preg_match('/\-(T|F|P)$/', $c, $m)) $doc_type = $m[1];
                                else $doc_type = 'T';
                            }
                            $doc_label = ($doc_type==='F') ? 'Factura' : (($doc_type==='P') ? 'Proforma' : 'Ticket');
                            $total_txt = html_entity_decode(wp_strip_all_tags(wc_price($order->get_total())), ENT_QUOTES, 'UTF-8');
                            $search_blob = trim(strtolower($code.' '.$doc_label.' '.$order->get_date_created()->date_i18n('Y-m-d H:i').' '.$total_txt.' '.$order->get_status()));
                            echo '<tr data-oid="'.esc_attr($oid).'" data-doctype="'.esc_attr($doc_type).'" data-search="'.esc_attr($search_blob).'"><td>'.esc_html($code).'</td><td>'.esc_html($order->get_date_created()->date_i18n('Y-m-d H:i')).'</td><td class="total-cell">'.esc_html($total_txt).'</td><td class="status-cell">'.esc_html($order->get_status()).'</td><td class="actions">'
                                . '<button class="btn" data-print="'.$oid.'">IMPRIMIR</button>'
                                . '<button class="btn" data-edit="'.$oid.'">MODIFICAR</button>'
                                . '<a class="btn" target="_blank" href="'.esc_url(add_query_arg(array(POS_TIENDA_QUERYVAR=>1,'order_id'=>$oid),home_url('/'))).'">EDITAR EN POS</a>'
                                . '<button class="btn bad" data-del="'.$oid.'">ELIMINAR</button>'
                                . '<button class="btn" data-gift="'.$oid.'">TICKET REGALO</button>'
                                . '</td></tr>';
                        }
                    } else { echo '<tr><td colspan="5">No hay pedidos POS.</td></tr>'; } ?>
                    </tbody>
                </table>
            </div>
        </section></main>
        <div class="modal" id="posModal">
            <div class="card">
              <header>Editar pedido <span id="mOrder"></span></header>
              <div class="body">
                <div class="grid header" id="mHead"><div>Producto</div><div>Precio</div><div>Uds</div><div>Total</div><div></div></div>
                <div id="mRows"></div>
                <div style="margin-top:10px">
                  <label for="mNotes" style="font-weight:600">Anotaciones</label>
                  <textarea id="mNotes" rows="3" placeholder="Observaciones de la venta"></textarea>
                </div>
              </div>
              <div class="footer"><button class="btn" id="mCancel">Cancelar</button><button class="btn primary" id="mSave">Guardar</button></div>
            </div>
        </div>
        <script>
        (function(){
            var modal=document.getElementById('posModal'); var mOrder=document.getElementById('mOrder'); var mRows=document.getElementById('mRows'); var mCancel=document.getElementById('mCancel'); var mSave=document.getElementById('mSave'); var table=document.getElementById('salesTable'); var currentId=null;
            var mNotes=document.getElementById('mNotes');
            var salesSearch = document.getElementById('salesSearch');
            var salesFilter = document.getElementById('salesFilter');

            function applySalesFilters(){
                if (!table) return;
                var term = (salesSearch && salesSearch.value) ? String(salesSearch.value).toLowerCase().trim() : '';
                var f = (salesFilter && salesFilter.value) ? String(salesFilter.value) : 'all';
                var rows = table.querySelectorAll('tbody tr');
                rows.forEach(function(row){
                    var ok = true;
                    if (f !== 'all'){
                        var dt = row.getAttribute('data-doctype') || '';
                        ok = (dt === f);
                    }
                    if (ok && term){
                        var txt = row.getAttribute('data-search') || (row.textContent || '');
                        txt = String(txt).toLowerCase();
                        ok = (txt.indexOf(term) !== -1);
                    }
                    row.style.display = ok ? '' : 'none';
                });
            }
            if (salesSearch) salesSearch.addEventListener('input', applySalesFilters);
            if (salesFilter) salesFilter.addEventListener('change', applySalesFilters);
            applySalesFilters();
            var toDelete = new Set();
            function onlyDigits(x){ return String(x||'').replace(/\D+/g,''); }
            function parseEU(v){ v=String(v||'').replace(/\u00A0/g,' ').trim().replace(/\./g,'').replace(',', '.'); var n=parseFloat(v); return isNaN(n)?0:n; }
            function openModal(id){
                id=onlyDigits(id); currentId=id; mOrder.textContent='#'+id; mRows.innerHTML='<div style="padding:12px">Cargando…</div>'; toDelete.clear(); modal.style.display='flex';
                fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>?action=pos_tienda_get_order&id='+encodeURIComponent(id), {credentials:'same-origin'}).then(r=>r.json()).then(function(d){
                    if(!d||!d.success){ mRows.innerHTML='<div style="padding:12px">No se pudo cargar</div>'; return; }
                    var rows=''; (d.data.items||[]).forEach(function(it){
                        var ptxt = it.price_text || String(it.price_raw||0).replace('.', ',');
                        rows += '<div class="row" style="display:grid;grid-template-columns:2fr 1fr .8fr 1fr .6fr;gap:8px;align-items:center;border:1px solid #e5efe5;border-radius:10px;padding:6px" data-item="'+it.item_id+'" data-pid="'+it.product_id+'"><div>'+it.name+' <small>'+ (it.sku||'') +'</small></div><div><input type="text" inputmode="decimal" value="'+ptxt+'"></div><div><input type="number" min="1" value="'+it.qty+'"></div><div class="line">'+it.total_text+'</div><div><button class="btn bad del">✕</button></div></div>';
                    }); mRows.innerHTML=rows||'<div style="padding:12px">Sin líneas</div>';
                    mNotes.value = (d.data && d.data.notes) ? d.data.notes : '';
                    mRows.querySelectorAll('.row').forEach(function(rw){
                        var inputs=rw.querySelectorAll('input');
                        function recalc(){ var p=parseEU(inputs[0].value); var q=parseInt(inputs[1].value||'1',10); if(!q||q<1)q=1; rw.querySelector('.line').textContent=(p*q).toLocaleString(window.POS_LOCALE,{style:'currency',currency:'EUR', useGrouping:true}); }
                        inputs[0].addEventListener('input',recalc); inputs[1].addEventListener('input',recalc);
                        var delBtn = rw.querySelector('.del');
                        delBtn.addEventListener('click', function(){
                            var iid = parseInt(rw.getAttribute('data-item'),10); if(iid){ toDelete.add(iid); }
                            rw.remove();
                        });
                    });
                }).catch(function(){ mRows.innerHTML='<div style="padding:12px">Error de conexión</div>'; });
            }
            function closeModal(){ modal.style.display='none'; currentId=null; toDelete.clear(); }
            document.addEventListener('click', function(e){
                var t=e.target; if(!t) return;
                if (t.matches('button[data-del]')){
                    var id=onlyDigits(t.getAttribute('data-del')); if(!confirm('Eliminar pedido POS (sin reponer stock)?'))return;
                    var fd=new FormData(); fd.append('action','pos_tienda_delete_order'); fd.append('id', id);
                    fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>',{method:'POST',credentials:'same-origin',body:fd}).then(r=>r.json()).then(function(d){ if(d&&d.success){ var tr=t.closest('tr'); if(tr) tr.remove(); } else { alert((d&&d.data&&d.data.message)?d.data.message:'No se pudo eliminar'); } }).catch(function(){ alert('Error de conexión'); });
                }
                if (t.matches('button[data-edit]')){ openModal(t.getAttribute('data-edit')); }
                if (t.matches('button[data-print]')){ var id=onlyDigits(t.getAttribute('data-print')); var url='<?php echo esc_js( admin_url('admin-ajax.php') ); ?>?action=pos_tienda_receipt&id='+encodeURIComponent(id)+'&render=1'; window.open(url,'_blank'); }
                if (t.matches('button[data-gift]')){ var id=onlyDigits(t.getAttribute('data-gift')); var url='<?php echo esc_js( admin_url('admin-ajax.php') ); ?>?action=pos_tienda_receipt&id='+encodeURIComponent(id)+'&render=1&gift=1'; window.open(url,'_blank'); }
            });
            mCancel.addEventListener('click', closeModal);
            mSave.addEventListener('click', function(){
                if(!currentId) return;
                var payload=[]; mRows.querySelectorAll('.row').forEach(function(rw){
                    var itemId=parseInt(rw.getAttribute('data-item'),10);
                    var pid=parseInt(rw.getAttribute('data-pid'),10)||0;
                    var inputs=rw.querySelectorAll('input');
                    var price=parseEU(inputs[0].value);
                    var qty=parseInt(inputs[1].value||'1',10); if(!qty||qty<1)qty=1;
                    payload.push({item_id:itemId, product_id:pid, price:price, qty:qty});
                });
                var fd=new FormData();
                fd.append('action','pos_tienda_update_order'); fd.append('id', currentId);
                fd.append('items', JSON.stringify(payload));
                fd.append('delete_items', JSON.stringify(Array.from(toDelete)));
                fd.append('notes', mNotes ? mNotes.value : '');
                fetch('<?php echo esc_js( admin_url('admin-ajax.php') ); ?>',{method:'POST',credentials:'same-origin',body:fd}).then(r=>r.json()).then(function(d){
                    if(d&&d.success){
                        var tr = table.querySelector('tr[data-oid="'+currentId+'"]');
                        if(tr){
                            var totalCell = tr.querySelector('.total-cell');
                            var statusCell = tr.querySelector('.status-cell');
                            if(d.data && d.data.new_total_text){ totalCell.textContent = d.data.new_total_text; }
                            else if(d.data && typeof d.data.new_total==='number'){ totalCell.textContent = d.data.new_total.toLocaleString(window.POS_LOCALE,{style:'currency',currency:'EUR'}); }
                            if(d.data && d.data.new_status){ statusCell.textContent = d.data.new_status; }
                        } else { try { location.reload(); } catch(e) {} }
                        closeModal();
                    } else { alert((d&&d.data&&d.data.message)?d.data.message:'No se guardó'); }
                }).catch(function(){ alert('Error de conexión'); });
            });
        })();
        </script>
    <?php endif; wp_footer(); ?></body></html><?php
    exit;
}


/* ===== INFORMES (excluye Proformas) ===== */
add_action('wp_ajax_pos_tienda_reports', function(){
    if (!current_user_can('manage_woocommerce') && !current_user_can('read')){
        wp_send_json_error(array('message'=>'Sin permisos'),403);
    }
    $from = isset($_POST['from']) ? sanitize_text_field($_POST['from']) : '';
    $to   = isset($_POST['to']) ? sanitize_text_field($_POST['to']) : '';
    $method = isset($_POST['method']) ? sanitize_text_field($_POST['method']) : 'all';
    $args = array('type'=>'shop_order','limit'=>-1,'orderby'=>'date','order'=>'DESC','status'=>'any','return'=>'objects','meta_query'=>array(array('key'=>'_pos_origin','value'=>'pos_tienda','compare'=>'=')));
    try { $q=new WC_Order_Query($args); $orders=$q->get_orders(); } catch (Throwable $e){ $orders=array(); }
    $rows=array(); $tAll=0; $tCash=0; $tCard=0; $cnt=0;
    $from_ts = $from ? strtotime($from.' 00:00:00') : 0;
    $to_ts   = $to   ? strtotime($to.' 23:59:59') : PHP_INT_MAX;
    foreach ($orders as $order){
        $ts = $order->get_date_created() ? $order->get_date_created()->getTimestamp() : 0;
        if ($ts<$from_ts || $ts>$to_ts) continue;
        $doc_type = $order->get_meta('_pos_doc_type', true);
        if ($doc_type === 'P') continue;
        $pm = $order->get_payment_method();
        $pm_meta = $order->get_meta('_pos_payment_method', true);
        if (!$pm && $pm_meta){ $pm = ($pm_meta==='card') ? 'pos-card' : (($pm_meta==='cash') ? 'pos-cash' : $pm_meta); }
        $label = ($pm==='pos-cash' || $pm_meta==='cash') ? 'Efectivo' : (($pm==='pos-card' || $pm_meta==='card') ? 'Tarjeta' : 'Desconocido');
        if ($method==='cash' && !($pm==='pos-cash' || $pm_meta==='cash')) continue;
        if ($method==='card' && !($pm==='pos-card' || $pm_meta==='card')) continue;
        $total = (float)$order->get_total();
        $code  = $order->get_meta('_pos_doc_code', true) ?: $order->get_id();
        $rows[] = array('code'=>$code,'date'=>$order->get_date_created()->date_i18n('Y-m-d H:i'),'method'=>$label,'total'=>$total,'status'=>$order->get_status());
        $tAll += $total; $cnt++;
        if ($pm==='pos-cash' || $pm_meta==='cash') $tCash += $total; elseif ($pm==='pos-card' || $pm_meta==='card') $tCard += $total;
    }
    wp_send_json_success(array('rows'=>$rows,'total'=>$tAll,'total_cash'=>$tCash,'total_card'=>$tCard,'count'=>$cnt));
});


/* ===== RECIBO / IMPRESIÓN (con anotaciones) ===== */
add_action('wp_ajax_pos_tienda_receipt', function(){
    $id = isset($_GET['id']) ? intval(preg_replace('/\D+/', '', (string)$_GET['id'])) : 0;
    $order = wc_get_order($id); if (!$order) wp_die('Pedido no encontrado');
    $gift = isset($_GET['gift']) && $_GET['gift'] === '1';
    $letter = $order->get_meta('_pos_doc_type', true) ?: 'T';
    $label  = $gift ? 'Ticket regalo' : ($letter==='F'?'Factura':($letter==='P'?'Proforma':'Ticket'));
    // Emisor: facturas/proformas vs tiques
    if (!$gift && $letter === 'T') {
        $issuer = (array) get_option('pos_tienda_issuer_tickets', array());
    } else {
        $issuer = (array) get_option('pos_tienda_issuer', array());
    }
    $notes  = trim((string) $order->get_meta('_pos_notes', true));
    // Impuestos (para desglosar totales)
    $tax_name_meta = (string) $order->get_meta('_pos_tax_name', true);
    $tax_rate_meta = $order->get_meta('_pos_tax_rate', true);

    $tax_name = sanitize_text_field( trim($tax_name_meta) );
    if ($tax_name === '') {
        $tax_name_opt = get_option('pos_tienda_tax_name', 'Impuesto');
        $tax_name = sanitize_text_field( is_string($tax_name_opt) ? $tax_name_opt : 'Impuesto' );
    }
    if ($tax_name === '') { $tax_name = 'Impuesto'; }

    $tax_rate = 0.0;
    if (is_numeric($tax_rate_meta)) {
        $tax_rate = (float) $tax_rate_meta;
    } else {
        $tax_rate_opt = get_option('pos_tienda_tax_rate', 0);
        $tax_rate = is_numeric($tax_rate_opt) ? (float) $tax_rate_opt : 0.0;
    }
    if ($tax_rate < 0) { $tax_rate = 0.0; }

    header('Content-Type: text/html; charset=UTF-8'); ?>
    <!doctype html><html lang="<?php echo esc_attr(pos_tienda_get_lang()); ?>"><head><meta charset="utf-8">
    <title><?php echo esc_html($label.' '.$order->get_meta('_pos_doc_code', true)); ?></title>
    <style>@media print{@page{margin:0}body{margin:10mm}a[href]:after{content:'' !important}}body{font:14px system-ui;margin:10mm;color:#000} table{width:100%;border-collapse:collapse} th,td{padding:6px;border-bottom:1px solid #ddd;text-align:left} tfoot td{font-weight:700}</style>
    <script>addEventListener('load',()=>{try{print()}catch(e){} setTimeout(()=>{try{close()}catch(e){}},600)})</script>
    </head><body>
    <?php pos_tienda_print_i18n_script(); ?>
    <div style="text-align:center">
        <?php if ($logo = get_site_icon_url(64)): ?><img src="<?php echo esc_url($logo); ?>" style="height:64px"><?php endif; ?>
        <?php if ($letter !== 'T'): ?><h1><?php bloginfo('name'); ?></h1><?php endif; ?>
        <div><?php echo esc_html($label); ?>: <?php echo esc_html($order->get_meta('_pos_doc_code', true)); ?></div>
        <?php
          $pm_meta = $order->get_meta('_pos_payment_method', true);
          $pm_wc   = $order->get_payment_method();
          $is_card = ($pm_meta === 'card') || ($pm_meta === 'pos-card') || ($pm_wc === 'pos-card');
          $pm_label = $is_card ? 'Tarjeta' : 'Efectivo';
          echo '<div>Método de pago: ' . esc_html($pm_label) . '</div>';
        // Mostrar efectivo entregado y cambio si hay efectivo (>0) y no es ticket regalo
        $cash_given  = (float) $order->get_meta('_pos_cash_given', true);
        if (!$gift && $cash_given > 0) {
          $cash_change = $order->get_meta('_pos_cash_change', true);
          $cash_change = is_numeric($cash_change) ? (float)$cash_change : 0.0;
          if ($cash_change <= 0) {
            $cash_change = max(0, $cash_given - (float)$order->get_total());
          }
          echo '<div>Cantidad entregada: ' . html_entity_decode(wp_strip_all_tags(wc_price($cash_given)), ENT_QUOTES, 'UTF-8') . '</div>';
          echo '<div>Cambio: ' . html_entity_decode(wp_strip_all_tags(wc_price($cash_change)), ENT_QUOTES, 'UTF-8') . '</div>';
        }

        ?>
<?php if (!$gift): ?>
  <?php if ($letter === 'T'): ?>
    <div style="margin:14px 0;text-align:center">
      <?php if (!empty($issuer['company'])): ?>
      <div><?php echo esc_html($issuer['company']); ?></div>
      <?php endif; ?>
      <?php if (!empty($issuer['name'])): ?>
      <div><?php echo esc_html($issuer['name']); ?></div>
      <?php endif; ?>
      <?php if (!empty($issuer['nif'])): ?>
      <div><?php echo esc_html($issuer['nif']); ?></div>
      <?php endif; ?>
      <?php if (!empty($issuer['address'])): ?>
      <div><?php echo esc_html($issuer['address']); ?></div>
      <?php endif; ?>
      <?php if (!empty($issuer['email'])): ?>
      <div><?php echo esc_html($issuer['email']); ?></div>
      <?php endif; ?>
      <?php if (!empty($issuer['phone'])): ?>
      <div><?php echo esc_html($issuer['phone']); ?></div>
      <?php endif; ?>
    </div>
  <?php else: ?>
    <div style="display:grid;grid-template-columns:1fr 1fr;gap:16px;margin:14px 0">
      <div style="border:1px solid #ddd;border-radius:10px;padding:10px">
        <div><?php echo esc_html($issuer['company'] ?? ''); ?></div>
        <div><?php echo esc_html($issuer['name'] ?? ''); ?></div>
        <div><?php echo esc_html($issuer['nif'] ?? ''); ?></div>
        <div><?php echo esc_html($issuer['address'] ?? ''); ?></div>
        <div><?php echo esc_html($issuer['email'] ?? ''); ?></div>
        <div><?php echo esc_html($issuer['phone'] ?? ''); ?></div>
      </div>
      <div style="border:1px solid #ddd;border-radius:10px;padding:10px">
        <h3 style="margin:0 0 6px">Cliente</h3>
        <div><?php echo esc_html( trim(
            ( $order->get_billing_company() ? $order->get_billing_company().' ' : '' )
            . $order->get_billing_first_name().' '.$order->get_billing_last_name()
        ) ); ?></div>
        <div><?php echo esc_html($order->get_meta('_billing_nif', true)); ?></div>
        <div><?php echo esc_html($order->get_billing_address_1()); ?></div>
        <div><?php echo esc_html($order->get_billing_email()); ?></div>
        <div><?php echo esc_html($order->get_billing_phone()); ?></div>
      </div>
    </div>
  <?php endif; ?>
<?php endif; ?>
</div>
        <div>Fecha: <?php echo esc_html($order->get_date_created()->date_i18n('Y-m-d')); ?></div>
    </div>
    <h2>Productos</h2>
    <?php if ($gift): ?>
    <table><thead><tr><th>Producto</th><th>SKU</th><th>Uds</th></tr></thead><tbody>
    <?php foreach ($order->get_items() as $item_id => $item){
        $qty  = max(1, (int) $item->get_quantity());
        $sku  = (string) $item->get_meta('_pos_sku', true);
        if ($sku === '') {
            $pid = (int) $item->get_product_id();
            if ($pid) { $sku = (string) get_post_meta($pid, '_sku', true); }
        }
    ?>
        <tr>
            <td><?php echo esc_html($item->get_name()); ?></td>
            <td><?php echo esc_html($sku); ?></td>
            <td><?php echo esc_html($qty); ?></td>
        </tr>
    <?php } ?>
    </tbody></table>
<?php else: ?>
    <table>
        <thead>
            <tr>
                <th>Producto</th>
                <th>SKU</th>
                <th>Precio</th>
                <th>Uds</th>
                <th>Dto %</th>
                <th>Dto</th>
                <th>Total</th>
            </tr>
        </thead>
        <tbody>
        <?php foreach ($order->get_items() as $item_id => $item){
                        $qty      = max(1, (int) $item->get_quantity());
            $sku      = (string) $item->get_meta('_pos_sku', true);
            if ($sku === '') {
                $pid = (int) $item->get_product_id();
                if ($pid) { $sku = (string) get_post_meta($pid, '_sku', true); }
            }

            $subtotal = (float) $item->get_subtotal();
            $tot      = (float) $item->get_total();
            $unit     = $qty > 0 ? $tot / $qty : 0;
            // Descuento guardado por el POS (si existe)
            $disc_pct_meta = $item->get_meta('_pos_disc_percent', true);
            $disc_pct      = is_numeric($disc_pct_meta) ? (float) $disc_pct_meta : 0.0;
            if ($disc_pct <= 0 && $subtotal > 0 && $tot < $subtotal) {
                $disc_pct = 100 - ($tot * 100 / $subtotal);
            }
            if ($disc_pct < 0)   $disc_pct = 0.0;
            if ($disc_pct > 100) $disc_pct = 100.0;
            $disc_amt_meta = $item->get_meta('_pos_disc_amount', true);
            $disc_amt      = is_numeric($disc_amt_meta) ? (float) $disc_amt_meta : 0.0;
            if ($disc_amt <= 0 && $subtotal > 0 && $tot < $subtotal) {
                $disc_amt = $subtotal - $tot;
            }
        ?>
            <tr>
                <td><?php echo esc_html($item->get_name()); ?></td>
                <td><?php echo esc_html($sku); ?></td>
                <td><?php echo wc_price($unit); ?></td>
                <td><?php echo esc_html($qty); ?></td>
                <td><?php echo $disc_pct > 0 ? esc_html( number_format_i18n( $disc_pct, 2 ) . '%' ) : ''; ?></td>
                <td><?php echo $disc_amt > 0 ? wc_price( $disc_amt ) : ''; ?></td>
                <td><?php echo wc_price($tot); ?></td>
            </tr>
        <?php } ?>
        </tbody>
        <tfoot>
<?php
                $total_incl = (float) $order->get_total();
                $total_sin  = $total_incl;
                $tax_amt    = 0.0;
                if ($tax_rate > 0) {
                    $div = 1 + ($tax_rate / 100);
                    if ($div != 0) {
                        $total_sin = $total_incl / $div;
                        $tax_amt   = $total_incl - $total_sin;
                    }
                }
                $total_sin = round($total_sin, 2);
                $tax_amt   = round($tax_amt, 2);
            ?>
            <tr>
                <td colspan="6" style="text-align:right">Total sin <?php echo esc_html($tax_name); ?></td>
                <td><?php echo wc_price($total_sin); ?></td>
            </tr>
            <tr>
                <td colspan="6" style="text-align:right"><?php echo esc_html($tax_name); ?></td>
                <td><?php echo wc_price($tax_amt); ?></td>
            </tr>
            <tr>
                <td colspan="6" style="text-align:right">Total <?php echo esc_html($tax_name); ?> incuido</td>
                <td><?php echo wc_price($total_incl); ?></td>
            </tr>
        </tfoot>
    </table>
<?php endif; ?>

    <?php if ($notes !== ''): ?>
      <div style="margin-top:10mm; border-top:1px dashed #ccc; padding-top:6px;">
        <strong>Anotaciones:</strong>
        <div><?php echo nl2br(esc_html($notes)); ?></div>
      </div>
    <?php endif; ?>
    </body></html><?php
    exit;
});


/* ===== OBTENER / EDITAR PEDIDO ===== */
add_action('wp_ajax_pos_tienda_get_order', function(){
    if (!current_user_can('manage_woocommerce') && !current_user_can('edit_shop_orders')) wp_send_json_error(array('message'=>'Sin permisos'),403);
    $id = isset($_GET['id']) ? intval(preg_replace('/\D+/', '', (string)$_GET['id'])) : 0;
    $order = wc_get_order($id); if (!$order) wp_send_json_error(array('message'=>'Pedido no encontrado'),404);
    $items = array();
    foreach ($order->get_items(array('line_item')) as $item_id=>$item){
        $prod = $item->get_product();
        $qty  = max(1,(int)$item->get_quantity());
        $tot  = (float)$item->get_total();
        $unit = $qty>0 ? $tot/$qty : 0;
        $items[] = array(
            'item_id'=>$item_id,
            'product_id'=>$prod ? $prod->get_id() : 0,
            'name'=>$item->get_name(),
            'sku'=>$prod ? $prod->get_sku() : '',
            'price_raw'=>round($unit,2),
            'price_text'=>number_format_i18n( $unit, 2 ),
            'qty'=>$qty,
            'total_text'=>html_entity_decode(wp_strip_all_tags(wc_price($unit*$qty)), ENT_QUOTES, 'UTF-8')
        );
    }
    $notes = (string) $order->get_meta('_pos_notes', true);
    wp_send_json_success(array('items'=>$items,'notes'=>$notes));
});

add_action('wp_ajax_pos_tienda_update_order', function(){
    if (!current_user_can('manage_woocommerce') && !current_user_can('edit_shop_orders')) wp_send_json_error(array('message'=>'Sin permisos'),403);
    $id = isset($_POST['id']) ? intval(preg_replace('/\D+/', '', (string)$_POST['id'])) : 0;
    $order = wc_get_order($id); if (!$order) wp_send_json_error(array('message'=>'Pedido no encontrado'),404);
    $list = isset($_POST['items']) ? json_decode(stripslashes($_POST['items']), true) : array();
    if (!is_array($list)) $list = array();
    $to_delete = isset($_POST['delete_items']) ? json_decode(stripslashes($_POST['delete_items']), true) : array();
    if (!is_array($to_delete)) $to_delete = array();
    $notes_raw = isset($_POST['notes']) ? wp_unslash($_POST['notes']) : '';
    $notes = is_string($notes_raw) ? sanitize_textarea_field($notes_raw) : '';

    $map = array(); foreach ($order->get_items(array('line_item')) as $item_id=>$item){ $map[$item_id] = $item; }

    foreach ($to_delete as $iid){
        $iid = intval($iid);
        if ($iid && isset($map[$iid])){
            $order->remove_item($iid);
            unset($map[$iid]);
        }
    }

    foreach ($list as $line){
        $iid = isset($line['item_id']) ? intval($line['item_id']) : 0;
        $pid = isset($line['product_id']) ? intval($line['product_id']) : 0;
        $qty = isset($line['qty']) ? max(1,intval($line['qty'])) : 1;
        $price = isset($line['price']) ? floatval($line['price']) : 0.0;
        if ($iid && isset($map[$iid])){
            $item = $map[$iid];
            $item->set_quantity($qty);
            $item->set_subtotal($price*$qty);
            $item->set_total($price*$qty);
            $item->save();
        } elseif ($pid){
            $product = wc_get_product($pid);
            if ($product){
                $item = new WC_Order_Item_Product();
                $item->set_product($product);
                $item->set_quantity($qty);
                $item->set_subtotal($price*$qty);
                $item->set_total($price*$qty);
                $order->add_item($item);
                $item->save();
            }
        }
    }

    $order->update_meta_data('_pos_notes', $notes);

    $order->calculate_taxes();
    $order->calculate_totals(true);
    $order->save();
    $plain_total = html_entity_decode(wp_strip_all_tags(wc_price($order->get_total())), ENT_QUOTES, 'UTF-8');
    wp_send_json_success(array('new_total'=>(float)$order->get_total(),'new_total_text'=>$plain_total,'new_status'=>$order->get_status()));
});


/* ===== ELIMINAR PEDIDO (sin reponer stock) ===== */
add_action('wp_ajax_pos_tienda_delete_order', function(){
    if (!current_user_can('delete_shop_orders') && !current_user_can('manage_ocommerce') && !current_user_can('administrator')){
    }
    if (!current_user_can('delete_shop_orders') && !current_user_can('manage_woocommerce') && !current_user_can('administrator')){
        wp_send_json_error(array('message'=>'Sin permisos'),403);
    }
    $id = isset($_POST['id']) ? intval(preg_replace('/\D+/', '', (string)$_POST['id'])) : 0;
    $order = wc_get_order($id); if (!$order) wp_send_json_error(array('message'=>'Pedido no encontrado'),404);
    $is_pos = ($order->get_meta('_pos_origin', true)==='pos_tienda') || (bool)$order->get_meta('_pos_doc_code', true);
    if (!$is_pos) wp_send_json_error(array('message'=>'Solo pedidos POS'),400);
    $cb = function(){ return false; }; add_filter('woocommerce_can_restore_order_stock', $cb, 9999);
    $ok=false; try{ $order->delete(true); $ok=true; }catch(Throwable $e){ $ok=false; }
    if (!$ok){ try{ if (function_exists('wc_delete_order')){ wc_delete_order($id,true); $ok=true; } } catch (Throwable $e){ $ok=false; } }
    if (!$ok){ wp_delete_post($id,true); $ok=(get_post_status($id)===false); }
    remove_filter('woocommerce_can_restore_order_stock', $cb, 9999);
    if ($ok) wp_send_json_success(array('deleted'=>true)); else wp_send_json_error(array('message'=>'No se pudo eliminar'),500);
});


/* ===== CREAR PEDIDO (Proforma sin stock) ===== */

/* ===== CREAR / ACTUALIZAR PEDIDO POS ===== */
add_action('wp_ajax_pos_tienda_make_order', function(){
    if (!( current_user_can('manage_woocommerce') || current_user_can('edit_shop_orders') || current_user_can('administrator') )){
        wp_send_json_error(array('message'=>'Sin permisos'), 403);
    }
    if (!class_exists('WC_Order')) {
        wp_send_json_error(array('message'=>'WooCommerce no está disponible'), 500);
    }

    $raw = isset($_POST['cart']) ? wp_unslash($_POST['cart']) : '[]';
    $cart = json_decode($raw, true);
    if (!is_array($cart)) {
        $cart = array();
    }
    if (!$cart) {
        wp_send_json_error(array('message'=>'Carrito vacío'), 400);
    }

    $doc = isset($_POST['doc']) ? sanitize_text_field(wp_unslash($_POST['doc'])) : 'T';

    $pay_raw = isset($_POST['pay_method']) ? wp_unslash($_POST['pay_method']) : '';
    $pay_l   = is_string($pay_raw) ? strtolower(trim($pay_raw)) : '';
    $pay     = ($pay_l === 'card') ? 'card' : 'cash';

    // ID de pedido a editar (0 = nuevo)
    $edit_id = 0;
    if (isset($_POST['order_id'])) {
        $edit_id = absint( preg_replace('/\D+/', '', (string) wp_unslash($_POST['order_id'])) );
    }

    $order   = null;
    $is_edit = false;
    if ($edit_id > 0) {
        $order = wc_get_order($edit_id);
        if ($order && $order->get_id()) {
            $origin = $order->get_meta('_pos_origin', true);
            if ($origin === 'pos_tienda') {
                $is_edit = true;
            }
        }
    }

    if (!$order || !$order->get_id() || !$is_edit) {
        // Crear pedido nuevo
        $order   = wc_create_order();
        $is_edit = false;
    } else {
        // Mantener el tipo de documento original si existe
        $existing_doc = $order->get_meta('_pos_doc_type', true);
        if ($existing_doc) {
            $doc = $existing_doc;
        }
        // Eliminar las líneas de producto para volver a generarlas
        foreach ($order->get_items('line_item') as $item_id => $item) {
            $order->remove_item($item_id);
        }
    }

    // Añadir líneas del carrito
    foreach ($cart as $c){
        if (!is_array($c)) {
            continue;
        }
        $pid   = isset($c['id'])   ? absint($c['id'])      : 0;
        $qty   = isset($c['qty'])  ? floatval($c['qty'])   : 0;
        $price = isset($c['price'])? floatval($c['price']) : 0;
        $disc  = isset($c['disc']) ? floatval($c['disc'])  : 0;
        $name  = isset($c['title'])? (string) $c['title']  : '';
        $sku   = isset($c['sku'])  ? (string) $c['sku']    : '';

        if ($qty <= 0) {
            continue;
        }

                $item = new WC_Order_Item_Product();
        if ($pid) {
            // Variaciones: usar WooCommerce para asignar bien IDs/atributos
            if (get_post_type($pid) === 'product_variation') {
                $product = wc_get_product($pid);
                if ($product) {
                    $item->set_product($product);
                    $item->set_name($product->get_name());
                } else {
                    $item->set_product_id($pid);
                    if ($name !== '') {
                        $item->set_name($name);
                    } else {
                        $t = get_the_title($pid);
                        if ($t) { $item->set_name($t); }
                    }
                }
            } else {
                // Simple: evitar cargar el producto completo (mejor rendimiento)
                $item->set_product_id($pid);
                if ($name !== '') {
                    $item->set_name($name);
                } else {
                    $t = get_the_title($pid);
                    if ($t) { $item->set_name($t); }
                }
            }
        } else {
            $item->set_name($name);
        }

        // Guardar SKU del POS para imprimir sin cargar el producto
        if ($sku !== '') {
            $item->update_meta_data('_pos_sku', sanitize_text_field($sku));
        }

$item->set_quantity($qty);
$disc_pct  = max(0, min(100, $disc));
$line_base = $price * $qty;
$line      = $line_base * (1 - $disc_pct / 100);
$item->set_subtotal($line_base);
$item->set_total($line);
if ($disc_pct > 0 && $line_base > $line) {
    $item->update_meta_data('_pos_disc_percent', $disc_pct);
    $item->update_meta_data('_pos_disc_amount', $line_base - $line);
}
$order->add_item($item);
}

    // Método de pago y estado
    $order->set_payment_method( $pay === 'card' ? 'pos-card' : 'pos-cash' );
    $order->update_meta_data('_pos_payment_method', $pay);
    // En ventas presenciales del POS, por defecto marcamos como completado (venta cerrada).
    // Proforma: la dejamos en espera para distinguirla (no descuenta stock y no se considera venta cerrada).
    $desired_status = ($doc === 'P') ? 'on-hold' : 'completed';
    if (!$is_edit || ! $order->get_status()) {
        $order->set_status($desired_status);
    } else {
        // Si se editó un pedido POS que quedó como "processing", lo normalizamos.
        $cur = (string) $order->get_status();
        if ($desired_status === 'completed' && in_array($cur, array('processing','pending','on-hold'), true)) {
            $order->set_status('completed');
        }
        if ($desired_status === 'on-hold' && $cur !== 'on-hold') {
            $order->set_status('on-hold');
        }
    }

    // Guardar efectivo entregado
    $cash_raw = isset($_POST['cash_given']) ? wp_unslash($_POST['cash_given']) : '';
    $cash_num = 0.0;
    if (is_string($cash_raw)) {
        $cash_s = preg_replace('/[^0-9,\.]/', '', $cash_raw);
        if (substr_count($cash_s, ',') === 1 && substr_count($cash_s, '.') >= 1) {
            $cash_s = str_replace('.', '', $cash_s);
        }
        $cash_s = str_replace(',', '.', $cash_s);
        $cash_num = floatval($cash_s);
    }
    if ($pay === 'card') {
        $cash_num = 0.0;
    }
    $order->update_meta_data('_pos_cash_given', $cash_num);

    // Datos de receptor (factura/proforma)
    if (isset($_POST['inv_company']) || isset($_POST['inv_name']) || isset($_POST['inv_nif']) || isset($_POST['inv_address']) || isset($_POST['inv_phone']) || isset($_POST['inv_email'])) {
        $inv_company = isset($_POST['inv_company']) ? sanitize_text_field( wp_unslash($_POST['inv_company']) ) : '';
        $inv_name    = isset($_POST['inv_name'])    ? sanitize_text_field( wp_unslash($_POST['inv_name']) )    : '';
        $inv_nif     = isset($_POST['inv_nif'])     ? sanitize_text_field( wp_unslash($_POST['inv_nif']) )     : '';
        $inv_addr    = isset($_POST['inv_address']) ? sanitize_text_field( wp_unslash($_POST['inv_address']) ) : '';
        $inv_phone   = isset($_POST['inv_phone'])   ? sanitize_text_field( wp_unslash($_POST['inv_phone']) )   : '';
        $inv_email   = isset($_POST['inv_email'])   ? sanitize_email( wp_unslash($_POST['inv_email']) )        : '';

        $order->set_billing_company( $inv_company );
        $order->set_billing_first_name( $inv_name );
        $order->set_billing_last_name( '' );
        $order->set_billing_address_1( $inv_addr );
        $order->set_billing_phone( $inv_phone );
        $order->set_billing_email( $inv_email );
        $order->update_meta_data('_billing_nif', $inv_nif );
    }

    // Notas internas del pedido POS
    $notes_raw = isset($_POST['notes']) ? wp_unslash($_POST['notes']) : '';
    $notes     = is_string($notes_raw) ? sanitize_textarea_field($notes_raw) : '';
    if ($notes !== '') {
        $order->update_meta_data('_pos_notes', $notes);
    }

    $order->update_meta_data('_pos_origin','pos_tienda');
    $order->update_meta_data('_pos_doc_type',$doc);

    // Marcar origen POS para que en la lista de pedidos aparezca "POS OfiSoftWeb" en la columna Origen.
    pos_tienda_mark_order_as_pos($order);

    // Marcar origen POS para que WooCommerce muestre "POS OfiSoftWeb" en "Origen".
    pos_tienda_mark_order_as_pos($order);

    // Guardar impuestos configurados (nombre y porcentaje) para este pedido
    $tax_name_opt = get_option('pos_tienda_tax_name', 'Impuesto');
    $tax_name_val = sanitize_text_field( is_string($tax_name_opt) ? $tax_name_opt : 'Impuesto' );
    if ($tax_name_val === '') { $tax_name_val = 'Impuesto'; }
    $tax_rate_opt = get_option('pos_tienda_tax_rate', 0);
    $tax_rate_val = is_numeric($tax_rate_opt) ? (float) $tax_rate_opt : 0.0;
    if ($tax_rate_val < 0) { $tax_rate_val = 0.0; }
    if (!$is_edit) {
        $order->update_meta_data('_pos_tax_name', $tax_name_val);
        $order->update_meta_data('_pos_tax_rate', $tax_rate_val);
    } else {
        $ex_name = $order->get_meta('_pos_tax_name', true);
        $ex_rate = $order->get_meta('_pos_tax_rate', true);
        if ($ex_name === '' || $ex_name === null) {
            $order->update_meta_data('_pos_tax_name', $tax_name_val);
        }
        if ($ex_rate === '' || $ex_rate === null) {
            $order->update_meta_data('_pos_tax_rate', $tax_rate_val);
        }
    }


    // Numeración de documento (solo pedidos nuevos)
    if (!$is_edit) {
        $year_now = (int) date('Y');
        $opt_year = (int) get_option('pos_tienda_docnum_year', $year_now);
        if ($opt_year !== $year_now){
            $opt_year = $year_now;
            update_option('pos_tienda_docnum_year', $opt_year, false);
            update_option('pos_tienda_docnum_counter_T', 0, false);
            update_option('pos_tienda_docnum_counter_P', 0, false);
            update_option('pos_tienda_docnum_counter_F', 0, false);
        }
        $doc_letter   = in_array($doc, array('T','P','F'), true) ? $doc : 'T';
        $counter_key  = 'pos_tienda_docnum_counter_' . $doc_letter;
        $opt_counter  = (int) get_option($counter_key, 0);
        $opt_counter++;
        update_option($counter_key, $opt_counter, false);
        $code = sprintf('%d-%02d-%s', $opt_counter, ((int) wp_date('y')), $doc_letter);
        $order->update_meta_data('_pos_doc_code', $code);
        $order->update_meta_data('_pos_doc_number', $code);
    }

    // Totales (evitar trabajo duplicado)
$order->calculate_totals(true);

// Calcular cambio (sin guardar dos veces)
try {
    $cash_given_meta = (float) $order->get_meta('_pos_cash_given', true);
    $total_now       = (float) $order->get_total();
    $change_calc     = ($cash_given_meta > 0) ? max(0, $cash_given_meta - $total_now) : 0.0;
    $order->update_meta_data('_pos_cash_change', $change_calc);
} catch (Throwable $e) {}

$order->save();

// Stock solo en pedidos nuevos y no proforma
    if (!$is_edit && $doc !== 'P') {
        if (function_exists('wc_reduce_stock_levels')){
            wc_reduce_stock_levels($order->get_id());
        } else {
            $order->reduce_order_stock();
        }
    }

    $url = add_query_arg(array(
        'action' => 'pos_tienda_receipt',
        'id'     => $order->get_id(),
        'render' => 1,
    ), admin_url('admin-ajax.php'));

    wp_send_json_success(array(
        'order_id'    => $order->get_id(),
        'receipt_url' => $url,
    ));
});





/* ===== ENUMERACIONES: LEER/RESETEAR ===== */
add_action('wp_ajax_pos_tienda_get_docnums', function(){
    if (!current_user_can('manage_woocommerce') && !current_user_can('edit_shop_orders')){
        wp_send_json_error(array('message'=>'Sin permisos'), 403);
    }
    $year_now = (int) date('Y');
    $opt_year = (int) get_option('pos_tienda_docnum_year', $year_now);
    $data = array(
        'year' => $opt_year,
        'counters' => array(
            'T' => (int) get_option('pos_tienda_docnum_counter_T', 0),
            'P' => (int) get_option('pos_tienda_docnum_counter_P', 0),
            'F' => (int) get_option('pos_tienda_docnum_counter_F', 0),
        ),
    );
    wp_send_json_success($data);
});

add_action('wp_ajax_pos_tienda_reset_docnums', function(){
    if (!current_user_can('manage_woocommerce') && !current_user_can('edit_shop_orders')){
        wp_send_json_error(array('message'=>'Sin permisos'), 403);
    }
    $changed = array();
    foreach (array('T','P','F') as $t){
        if (isset($_POST[$t]) && $_POST[$t]){
            update_option('pos_tienda_docnum_counter_'.$t, 0, false);
            $changed[] = $t;
        }
    }
    $resp = array(
        'changed' => $changed,
        'counters' => array(
            'T' => (int) get_option('pos_tienda_docnum_counter_T', 0),
            'P' => (int) get_option('pos_tienda_docnum_counter_P', 0),
            'F' => (int) get_option('pos_tienda_docnum_counter_F', 0),
        ),
    );
    wp_send_json_success($resp);
});

/* ===== CARGAR PEDIDO POS PARA EDITAR ===== */
add_action('wp_ajax_pos_tienda_load_order', function(){
    if (!( current_user_can('manage_woocommerce') || current_user_can('edit_shop_orders') || current_user_can('administrator') )){
        wp_send_json_error(array('message'=>'Sin permisos'), 403);
    }
    if (!class_exists('WC_Order')) {
        wp_send_json_error(array('message'=>'WooCommerce no está disponible'), 500);
    }

    $order_id = 0;
    if (isset($_POST['order_id'])) {
        $order_id = absint( preg_replace('/\D+/', '', (string) wp_unslash($_POST['order_id'])) );
    } elseif (isset($_GET['order_id'])) {
        $order_id = absint( preg_replace('/\D+/', '', (string) wp_unslash($_GET['order_id'])) );
    }

    if ($order_id <= 0) {
        wp_send_json_error(array('message' => 'ID de pedido no válido'), 400);
    }

    $order = wc_get_order($order_id);
    if (!$order || !$order->get_id()) {
        wp_send_json_error(array('message' => 'Pedido no encontrado'), 404);
    }

    $items_out = array();
    foreach ($order->get_items('line_item') as $item_id => $item){
        $product  = $item->get_product();
        $pid      = $item->get_product_id();
        $qty      = $item->get_quantity();
        $name     = $item->get_name();
        $sku      = $product ? (string) $product->get_sku() : '';
        $subtotal = (float) $item->get_subtotal();
        $total    = (float) $item->get_total();
        $disc     = 0.0;
        if ($subtotal > 0 && $total >= 0 && $total <= $subtotal) {
            $disc = 100 - ($total * 100 / $subtotal);
            if ($disc < 0)   $disc = 0.0;
            if ($disc > 100) $disc = 100.0;
        }
        $price = $qty > 0 ? ($total / $qty) : $total;
        $stock = null;
        if ($product && $product->managing_stock()){
            $stock = (float) $product->get_stock_quantity();
        }

        $items_out[] = array(
            'id'    => (int) $pid,
            'title' => $name,
            'sku'   => $sku,
            'price' => (float) $price,
            'qty'   => (float) $qty,
            'disc'  => (float) $disc,
            'stock' => $stock,
        );
    }

    $billing = array(
        'company' => $order->get_billing_company(),
        'name'    => $order->get_billing_first_name(),
        'nif'     => $order->get_meta('_billing_nif', true),
        'address' => $order->get_billing_address_1(),
        'phone'   => $order->get_billing_phone(),
        'email'   => $order->get_billing_email(),
    );
    $doc   = $order->get_meta('_pos_doc_type', true);
    if (!$doc) {
        $doc = 'T';
    }
    $notes = $order->get_meta('_pos_notes', true);

    wp_send_json_success(array(
        'items'   => $items_out,
        'billing' => $billing,
        'doc'     => $doc,
        'notes'   => $notes,
    ));
});

/* ===== GUARDAR/LEER EMISOR ===== */
add_action('wp_ajax_pos_tienda_save_issuer', function(){
    if (!current_user_can('manage_woocommerce') && !current_user_can('edit_shop_orders')){
        wp_send_json_error(array('message'=>'Sin permisos'), 403);
    }
    // Datos emisor para facturas y proformas
    $data_main = array(
        'company' => sanitize_text_field(isset($_POST['company']) ? $_POST['company'] : ''),
        'name'    => sanitize_text_field(isset($_POST['name']) ? $_POST['name'] : ''),
        'nif'     => sanitize_text_field(isset($_POST['nif']) ? $_POST['nif'] : ''),
        'address' => sanitize_text_field(isset($_POST['address']) ? $_POST['address'] : ''),
        'email'   => sanitize_email(isset($_POST['email']) ? $_POST['email'] : ''),
        'phone'   => sanitize_text_field(isset($_POST['phone']) ? $_POST['phone'] : ''),
    );
    // Datos emisor solo para tiques
    $data_tickets = array(
        'company' => sanitize_text_field(isset($_POST['company_t']) ? $_POST['company_t'] : ''),
        'name'    => sanitize_text_field(isset($_POST['name_t']) ? $_POST['name_t'] : ''),
        'nif'     => sanitize_text_field(isset($_POST['nif_t']) ? $_POST['nif_t'] : ''),
        'address' => sanitize_text_field(isset($_POST['address_t']) ? $_POST['address_t'] : ''),
        'email'   => sanitize_email(isset($_POST['email_t']) ? $_POST['email_t'] : ''),
        'phone'   => sanitize_text_field(isset($_POST['phone_t']) ? $_POST['phone_t'] : ''),
    );
    update_option('pos_tienda_issuer', $data_main, false);
    update_option('pos_tienda_issuer_tickets', $data_tickets, false);
    wp_send_json_success(array('message'=>'OK','issuer'=>$data_main,'issuer_tickets'=>$data_tickets));
});
add_action('wp_ajax_pos_tienda_save_tax', function(){
    if (!current_user_can('manage_woocommerce') && !current_user_can('edit_shop_orders')){
        wp_send_json_error(array('message'=>'Sin permisos'), 403);
    }
    $name_raw = isset($_POST['tax_name']) ? wp_unslash($_POST['tax_name']) : '';
    $name = is_string($name_raw) ? sanitize_text_field($name_raw) : '';
    if ($name === '') { $name = 'Impuesto'; }

    $rate_raw = isset($_POST['tax_rate']) ? wp_unslash($_POST['tax_rate']) : '';
    $rate_num = 0.0;
    if (is_string($rate_raw) || is_numeric($rate_raw)) {
        $s = is_string($rate_raw) ? $rate_raw : (string) $rate_raw;
        $s = trim($s);
        $s = str_replace('%', '', $s);
        $s = preg_replace('/[^0-9,\.]/', '', $s);
        if (strpos($s, ',') !== false) {
            $s = str_replace('.', '', $s);
            $s = str_replace(',', '.', $s);
        }
        $rate_num = floatval($s);
    }
    if ($rate_num < 0) { $rate_num = 0.0; }

    update_option('pos_tienda_tax_name', $name, false);
    update_option('pos_tienda_tax_rate', $rate_num, false);

    wp_send_json_success(array('message'=>'OK','tax_name'=>$name,'tax_rate'=>$rate_num));
});
add_action('wp_ajax_pos_tienda_get_settings', function(){
    // Emisor facturas/proformas
    $issuer_opt = (array) get_option('pos_tienda_issuer', array());
    $issuer = wp_parse_args($issuer_opt, array(
        'company' => 'OfiSoftWeb',
        'name'    => '',
        'nif'     => '',
        'address' => '',
        'email'   => 'tienda@OfiSoftWeb.com',
        'phone'   => '',
    ));
    // Emisor solo para tiques
    $issuer_tickets_opt = (array) get_option('pos_tienda_issuer_tickets', array());
    $issuer_tickets = wp_parse_args($issuer_tickets_opt, array(
        'company' => 'OfiSoftWeb',
        'name'    => '',
        'nif'     => '',
        'address' => '',
        'email'   => 'tienda@OfiSoftWeb.com',
        'phone'   => '',
    ));
    $tax_name_opt = get_option('pos_tienda_tax_name', 'Impuesto');
    $tax_name = sanitize_text_field( is_string($tax_name_opt) ? $tax_name_opt : 'Impuesto' );
    if ($tax_name === '') { $tax_name = 'Impuesto'; }
    $tax_rate_opt = get_option('pos_tienda_tax_rate', 0);
    $tax_rate = is_numeric($tax_rate_opt) ? (float) $tax_rate_opt : 0.0;
    if ($tax_rate < 0) { $tax_rate = 0.0; }
    wp_send_json_success(array('issuer'=>$issuer,'issuer_tickets'=>$issuer_tickets,'tax_name'=>$tax_name,'tax_rate'=>$tax_rate));
});


/* ===== UI MODAL EMISOR ===== */
if (!function_exists('pos_tienda_render_pos_settings')){
function pos_tienda_render_pos_settings(){
    $is_pos_route = (isset($_GET['POS_TIENDA']) && $_GET['POS_TIENDA'])
                 || (defined('POS_TIENDA_QUERYVAR') && isset($_GET[POS_TIENDA_QUERYVAR]) && $_GET[POS_TIENDA_QUERYVAR])
                 || (defined('POS_TIENDA_SALES_QUERYVAR') && isset($_GET[POS_TIENDA_SALES_QUERYVAR]) && $_GET[POS_TIENDA_SALES_QUERYVAR])
                 || (defined('POS_TIENDA_REPORTS_QUERYVAR') && isset($_GET[POS_TIENDA_REPORTS_QUERYVAR]) && $_GET[POS_TIENDA_REPORTS_QUERYVAR]);
    if (!$is_pos_route) return;
    $ajax = esc_url( admin_url('admin-ajax.php') );
    ?>
<div id="issuerModal" style="display:none;position:fixed;inset:0;background:rgba(0,0,0,.45);align-items:center;justify-content:center;z-index:9999">
  <div style="background:#fff;padding:16px;border-radius:12px;min-width:340px;max-width:720px">
    <h2 style="margin:0 0 8px">Datos del emisor</h2>
    <div style="display:grid;grid-template-columns:1fr 1fr;gap:16px">
      <div>
        <h3 style="margin:0 0 4px;font-size:14px">Facturas y proformas</h3>
        <div class="grid">
          <label>Empresa   <input id="emiCompany" type="text" placeholder="OfiSoftWeb"></label>
          <label>Nombre   <input id="emiName" type="text"></label>
          <label>NIF       <input id="emiNif" type="text"></label>
          <label>Dirección <input id="emiAddress" type="text"></label>
          <label>Email     <input id="emiEmail" type="email" placeholder="tienda@OfiSoftWeb.com"></label>
          <label>Teléfono  <input id="emiPhone" type="text"></label>
        </div>
      </div>
      <div>
        <h3 style="margin:0 0 4px;font-size:14px">Tiques</h3>
        <div class="grid">
          <label>Empresa   <input id="emiTCompany" type="text" placeholder="OfiSoftWeb"></label>
          <label>Nombre   <input id="emiTName" type="text"></label>
          <label>NIF       <input id="emiTNif" type="text"></label>
          <label>Dirección <input id="emiTAddress" type="text"></label>
          <label>Email     <input id="emiTEmail" type="email" placeholder="tienda@OfiSoftWeb.com"></label>
          <label>Teléfono  <input id="emiTPhone" type="text"></label>
        </div>
      </div>
    </div>
    <div style="display:flex;gap:8px;justify-content:flex-end;margin-top:12px">
      <button class="btn" id="issuerCancel">Cancelar</button>
      <button class="btn" id="issuerSave">Guardar</button>
    </div>
  </div>
</div>
<script>
(function(){
  var modal = document.getElementById('issuerModal');
  var cancel = document.getElementById('issuerCancel');
  var save = document.getElementById('issuerSave');
  if (cancel) cancel.addEventListener('click', function(){ modal.style.display='none'; });

  if (save) save.addEventListener('click', function(){
    var fd = new FormData();
    fd.append('action','pos_tienda_save_issuer');
    // Facturas / proformas
    fd.append('company', (document.getElementById('emiCompany')||{}).value||'');
    fd.append('name',    (document.getElementById('emiName')||{}).value||'');
    fd.append('nif',     (document.getElementById('emiNif')||{}).value||'');
    fd.append('address', (document.getElementById('emiAddress')||{}).value||'');
    fd.append('email',   (document.getElementById('emiEmail')||{}).value||'');
    fd.append('phone',   (document.getElementById('emiPhone')||{}).value||'');
    // Tiques
    fd.append('company_t', (document.getElementById('emiTCompany')||{}).value||'');
    fd.append('name_t',    (document.getElementById('emiTName')||{}).value||'');
    fd.append('nif_t',     (document.getElementById('emiTNif')||{}).value||'');
    fd.append('address_t', (document.getElementById('emiTAddress')||{}).value||'');
    fd.append('email_t',   (document.getElementById('emiTEmail')||{}).value||'');
    fd.append('phone_t',   (document.getElementById('emiTPhone')||{}).value||'');

    fetch('<?php echo $ajax; ?>',{method:'POST',credentials:'same-origin',body:fd})
      .then(r=>r.json()).then(function(d){
        if (d && d.success){ alert('Datos del emisor guardados'); modal.style.display='none'; }
        else { alert((d&&d.data&&d.data.message)||'No se pudo guardar'); }
      });
  });
})();
</script>
</script>
    <?php
}}
add_action('wp_footer','pos_tienda_render_pos_settings');

/* ===== UI MODAL RESET NUMERACIÓN ===== */
if (!function_exists('pos_tienda_render_docnum_tools')){
function pos_tienda_render_docnum_tools(){
    $is_pos_route = (isset($_GET['POS_TIENDA']) && $_GET['POS_TIENDA'])
                 || (defined('POS_TIENDA_QUERYVAR') && isset($_GET[POS_TIENDA_QUERYVAR]) && $_GET[POS_TIENDA_QUERYVAR])
                 || (defined('POS_TIENDA_SALES_QUERYVAR') && isset($_GET[POS_TIENDA_SALES_QUERYVAR]) && $_GET[POS_TIENDA_SALES_QUERYVAR])
                 || (defined('POS_TIENDA_REPORTS_QUERYVAR') && isset($_GET[POS_TIENDA_REPORTS_QUERYVAR]) && $_GET[POS_TIENDA_REPORTS_QUERYVAR]);
    if (!$is_pos_route) return;
    $ajax = esc_url( admin_url('admin-ajax.php') );
    ?>
<div id="docnumModal" style="display:none;position:fixed;inset:0;background:rgba(0,0,0,.45);align-items:center;justify-content:center;z-index:9999">
  <div style="background:#fff;padding:16px;border-radius:12px;min-width:340px;max-width:560px">
    <h2 style="margin:0 0 8px">Resetear numeración</h2>
    <p style="margin:0 0 8px">Año actual: <strong id="curYear">—</strong></p>
    <div class="grid">
      <label><input id="resetT" type="checkbox"> Ticket (T) — actual: <strong id="curT">0</strong></label>
      <label><input id="resetP" type="checkbox"> Proforma (P) — actual: <strong id="curP">0</strong></label>
      <label><input id="resetF" type="checkbox"> Factura  (F) — actual: <strong id="curF">0</strong></label>
    </div>
    <div style="display:flex;gap:8px;justify-content:flex-end;margin-top:12px">
      <button class="btn" id="docnumCancel">Cancelar</button>
      <button class="btn" id="docnumSave">Guardar</button>
    </div>
  </div>
</div>
<script>
(function(){
  var modal = document.getElementById('docnumModal');
  var cancel = document.getElementById('docnumCancel');
  var save = document.getElementById('docnumSave');
  function q(id){ return document.getElementById(id); }
  if (cancel) cancel.addEventListener('click', function(){ modal.style.display='none'; });

  // Abrir modal: lee contadores actuales
  var btnDoc = document.getElementById('btnDocnum');
  if (btnDoc){
    btnDoc.addEventListener('click', function(){
      var fd = new FormData(); fd.append('action','pos_tienda_get_docnums');
      fetch('<?php echo $ajax; ?>',{method:'POST',credentials:'same-origin',body:fd})
        .then(r=>r.json()).then(function(d){
          if (!d || !d.success){ alert('No se pudo leer la numeración'); return; }
          var x = d.data || {};
          (q('curYear')||{}).textContent = x.year || '';
          (q('curT')||{}).textContent    = (x.counters&&x.counters.T)||0;
          (q('curP')||{}).textContent    = (x.counters&&x.counters.P)||0;
          (q('curF')||{}).textContent    = (x.counters&&x.counters.F)||0;
          if (q('resetT')) q('resetT').checked = false;
          if (q('resetP')) q('resetP').checked = false;
          if (q('resetF')) q('resetF').checked = false;
          modal.style.display='flex';
        });
    });
  }

  if (save){
    save.addEventListener('click', function(){
      var fd = new FormData();
      fd.append('action','pos_tienda_reset_docnums');
      if ((q('resetT')||{}).checked) fd.append('T','1');
      if ((q('resetP')||{}).checked) fd.append('P','1');
      if ((q('resetF')||{}).checked) fd.append('F','1');
      fetch('<?php echo $ajax; ?>',{method:'POST',credentials:'same-origin',body:fd})
        .then(r=>r.json()).then(function(d){
          if (d && d.success){
            var c = d.data && d.data.counters || {};
            (q('curT')||{}).textContent = c.T||0;
            (q('curP')||{}).textContent = c.P||0;
            (q('curF')||{}).textContent = c.F||0;
            alert('Numeración actualizada');
            modal.style.display='none';
          } else {
            alert((d&&d.data&&d.data.message)||'No se pudo actualizar');
          }
        });
    });
  }
  if (typeof POS_EDIT_ORDER_ID !== 'undefined' && POS_EDIT_ORDER_ID > 0){ loadOrderForEdit(POS_EDIT_ORDER_ID); }
})();
</script>
    <?php
}}
add_action('wp_footer','pos_tienda_render_docnum_tools');




/* ===== CLIENTES POS: guardar ===== */
add_action('wp_ajax_pos_tienda_save_client', function(){
    if (!( current_user_can('manage_woocommerce') || current_user_can('edit_shop_orders') || current_user_can('administrator') )){
        wp_send_json_error(array('message'=>'Sin permisos'), 403);
    }

    $clients = get_option('pos_tienda_clients', array());
    if (!is_array($clients)) {
        $clients = array();
    }

    // ID recibido (0 = nuevo cliente)
    $id = 0;
    if (isset($_POST['id'])) {
        $id = absint( preg_replace('/\D+/', '', (string) wp_unslash($_POST['id'])) );
    }

    $name    = isset($_POST['name'])    ? sanitize_text_field(wp_unslash($_POST['name']))    : '';
    $company = isset($_POST['company']) ? sanitize_text_field(wp_unslash($_POST['company'])) : '';
    $nif     = isset($_POST['nif'])     ? sanitize_text_field(wp_unslash($_POST['nif']))     : '';
    $address = isset($_POST['address']) ? sanitize_text_field(wp_unslash($_POST['address'])) : '';
    $phone   = isset($_POST['phone'])   ? sanitize_text_field(wp_unslash($_POST['phone']))   : '';
    $email   = isset($_POST['email'])   ? sanitize_email(wp_unslash($_POST['email']))        : '';
    $notes   = isset($_POST['notes'])   ? sanitize_textarea_field(wp_unslash($_POST['notes'])) : '';

    if ($id <= 0) {
        // Nuevo cliente
        $max_id = 0;
        foreach ($clients as $c) {
            if (is_array($c) && isset($c['id'])) {
                $max_id = max($max_id, (int) $c['id']);
            }
        }
        $id = $max_id + 1;
        $clients[] = array(
            'id'      => $id,
            'name'    => $name,
            'company' => $company,
            'nif'     => $nif,
            'address' => $address,
            'phone'   => $phone,
            'email'   => $email,
            'notes'   => $notes,
        );
    } else {
        // Actualizar cliente existente
        $found = false;
        foreach ($clients as $k => $c) {
            if (is_array($c) && isset($c['id']) && (int) $c['id'] === $id) {
                $clients[$k] = array(
                    'id'      => $id,
                    'name'    => $name,
                    'company' => $company,
                    'nif'     => $nif,
                    'address' => $address,
                    'phone'   => $phone,
                    'email'   => $email,
                    'notes'   => $notes,
                );
                $found = true;
                break;
            }
        }
        if (!$found) {
            $clients[] = array(
                'id'      => $id,
                'name'    => $name,
                'company' => $company,
                'nif'     => $nif,
                'address' => $address,
                'phone'   => $phone,
                'email'   => $email,
                'notes'   => $notes,
            );
        }
    }

    $clients = array_values($clients);
    update_option('pos_tienda_clients', $clients, false);

    wp_send_json_success(array(
        'id'      => $id,
        'name'    => $name,
        'company' => $company,
        'nif'     => $nif,
        'address' => $address,
        'phone'   => $phone,
        'email'   => $email,
        'notes'   => $notes,
    ));
});

/* ===== CLIENTES POS: eliminar ===== */
add_action('wp_ajax_pos_tienda_delete_client', function(){
    if (!( current_user_can('manage_woocommerce') || current_user_can('edit_shop_orders') || current_user_can('administrator') )){
        wp_send_json_error(array('message'=>'Sin permisos'), 403);
    }

    $id = 0;
    if (isset($_POST['id'])) {
        $id = absint( preg_replace('/\D+/', '', (string) wp_unslash($_POST['id'])) );
    }
    if ($id <= 0) {
        wp_send_json_error(array('message' => 'ID de cliente no válido'), 400);
    }

    $clients = get_option('pos_tienda_clients', array());
    if (!is_array($clients)) {
        $clients = array();
    }

    $new = array();
    foreach ($clients as $c) {
        if (!is_array($c)) {
            continue;
        }
        if (isset($c['id']) && (int) $c['id'] === $id) {
            continue;
        }
        $new[] = $c;
    }

    $new = array_values($new);
    update_option('pos_tienda_clients', $new, false);

    wp_send_json_success(array('deleted' => true));
});

/* ===== CLIENTES POS: buscar ===== */
add_action('wp_ajax_pos_tienda_search_clients', function(){
    if (!( current_user_can('manage_woocommerce') || current_user_can('edit_shop_orders') || current_user_can('administrator') )){
        wp_send_json_error(array('message'=>'Sin permisos'), 403);
    }

    $q = '';
    if (isset($_POST['q'])) {
        $q = sanitize_text_field( wp_unslash( $_POST['q'] ) );
    }
    if ($q === '') {
        wp_send_json_success( array( 'items' => array() ) );
    }

    $clients = get_option('pos_tienda_clients', array());
    if (!is_array($clients)) {
        $clients = array();
    }

    $q_low = function_exists('mb_strtolower') ? mb_strtolower($q) : strtolower($q);

    $out = array();
    foreach ($clients as $c){
        if (!is_array($c)) continue;

        $id      = isset($c['id'])      ? (int) $c['id']      : 0;
        $name    = isset($c['name'])    ? (string) $c['name']    : '';
        $company = isset($c['company']) ? (string) $c['company'] : '';
        $nif     = isset($c['nif'])     ? (string) $c['nif']     : '';
        $address = isset($c['address']) ? (string) $c['address'] : '';
        $phone   = isset($c['phone'])   ? (string) $c['phone']   : '';
        $email   = isset($c['email'])   ? (string) $c['email']   : '';

        $haystack = $id.' '.$name.' '.$company.' '.$nif.' '.$address.' '.$phone.' '.$email;
        $hay_low = function_exists('mb_strtolower') ? mb_strtolower($haystack) : strtolower($haystack);

        if (strpos($hay_low, $q_low) === false) {
            continue;
        }

        $out[] = array(
            'id'      => $id,
            'name'    => $name,
            'company' => $company,
            'nif'     => $nif,
            'address' => $address,
            'phone'   => $phone,
            'email'   => $email,
        );
        if (count($out) >= 25) {
            break;
        }
    }

    wp_send_json_success( array( 'items' => $out ) );
});




// ===== DESACTIVAR EMAILS PARA PEDIDOS POS (para acelerar la generación del documento) =====
if ( ! function_exists( 'pos_tienda_is_pos_order' ) ) {
    function pos_tienda_is_pos_order( $order ) {
        if ( ! $order instanceof WC_Order ) {
            return false;
        }
        $origin   = $order->get_meta( '_pos_origin', true );
        $doc_code = $order->get_meta( '_pos_doc_code', true );

        if ( $origin === 'pos_tienda' ) {
            return true;
        }
        if ( ! empty( $doc_code ) ) {
            return true;
        }
        return false;
    }
}

if ( ! function_exists( 'pos_tienda_maybe_disable_email' ) ) {
    function pos_tienda_maybe_disable_email( $enabled, $order ) {
        if ( ! $enabled ) {
            return $enabled;
        }
        if ( ! $order instanceof WC_Order ) {
            return $enabled;
        }
        if ( pos_tienda_is_pos_order( $order ) ) {
            // Desactivar emails de WooCommerce para pedidos generados desde el POS
            return false;
        }
        return $enabled;
    }
}

add_filter( 'woocommerce_email_enabled_new_order', 'pos_tienda_maybe_disable_email', 20, 2 );
add_filter( 'woocommerce_email_enabled_customer_processing_order', 'pos_tienda_maybe_disable_email', 20, 2 );
add_filter( 'woocommerce_email_enabled_customer_completed_order', 'pos_tienda_maybe_disable_email', 20, 2 );
add_filter( 'woocommerce_email_enabled_customer_invoice', 'pos_tienda_maybe_disable_email', 20, 2 );

?>
